require "animate"
require "utils"

require "view/hand"
require "view/stat"

local BoardShortcuts = {'q', 'w', 'e', 'r', 't'}
local BoardShortcutsPositions = {q = 1, w = 2, e = 3, r = 4, t = 5, y = 6}
local spellSlotPlace = 6

local Elements = {"water", "fire", "air", "earth", "life", "death"}

local function elementXPosition(count) return 46 + count * 40 end

local function drawCardArea(game, player, y, animationDirection, events)
    local card_width = 95

    local showSpellSlot = not game.requireSpellSlot and player ==
                              game.currentPlayer() and
                              game.cardIsInHand(player, player.selectedCard) and
                              not player.played and player.selectedCard.type ==
                              "spell"
    local spellSlotX, spellSlotY, spellSlotWidth, spellSlotHeight =
        3 + 2 * card_width + 10, y + animationDirection * card_width / 2,
        card_width - 20, card_width - 20

    for place = 1, 6 do
        local x = 3 + (place - 1) * card_width

        if place == 6 then x, y = spellSlotX - 10, spellSlotY end

        local card = player.board[place]
        if card ~= nil then
            local yAnim = Animate(card.attacking, AttackEasing)
            local opacityAnim = 1 - Animate(card.dying, DeathEasing)

            if ClickedInside(x, y, card_width, card_width) then
                table.insert(events, function(game)
                    game.selectCard(card)
                end)
            end

            DrawCard(events, game, card, x, y + yAnim * animationDirection,
                     opacityAnim)

        elseif player == game.currentPlayer() and
            game.cardIsInHand(player, player.selectedCard) and not player.played and
            BoardShortcuts[place] then
            love.graphics.printf(BoardShortcuts[place], x, y + 25,
                                 card_width + 2, "center")
            if ClickedInside(x, y, card_width, card_width) then
                table.insert(events, function(game)
                    game.playCard(place)
                end)
            end
        end
    end

    if showSpellSlot then
        love.graphics.push("all")

        local drawBox = function(style)
            love.graphics.rectangle(style, spellSlotX, spellSlotY,
                                    spellSlotWidth, spellSlotHeight)
        end
        love.graphics.setColor(0, 0, 0)
        drawBox("fill")
        love.graphics.setColor(0xFF, 0xFF, 0xFF)
        drawBox("line")
        love.graphics.printf("y", spellSlotX, spellSlotY + spellSlotHeight / 3,
                             spellSlotWidth, "center")

        if ClickedInside(spellSlotX, spellSlotY, spellSlotWidth, spellSlotHeight) then
            table.insert(events,
                         function(game) game.playCard(spellSlotPlace) end)
        end
        love.graphics.pop()
    end
end

local function drawPlayersElements(events, player, y)
    -- Draw from right to left to avoid drawing over animations
    for i = 1, #Elements do
        local position = #Elements + 1 - i
        local element = Elements[position]
        DrawElement(events, element, position, y, player[element])
    end
end

function DrawBoard(game, assets, events)
    love.graphics.push()

    local x_offset = 15
    local y_offset = 15
    love.graphics.translate(x_offset, y_offset)

    love.graphics.draw(assets.background, 0, 0)

    local player1_y = 25
    local player2_y = 265

    -- drawPlayer(game.currentPlayer(), true)
    -- drawPlayer(game.opponent(game.currentPlayer()), false)

    if game.player1() == game.currentPlayer() then
        love.graphics.push("all")
        drawPlayersElements(events, game.player1(), player1_y)
        love.graphics.setColor(0xFF, 0xFF, 0xFF, 0.6);
        drawPlayersElements({}, game.player2(), player2_y)

        drawCardArea(game, game.player2(), 151, -1, events)
        drawCardArea(game, game.player1(), 62, 1, events)

        love.graphics.pop()
    else
        love.graphics.push("all")
        drawPlayersElements(events, game.player2(), player2_y)
        love.graphics.setColor(0xFF, 0xFF, 0xFF, 0.6);
        drawPlayersElements({}, game.player1(), player1_y)

        drawCardArea(game, game.player1(), 62, 1, events)
        drawCardArea(game, game.player2(), 151, -1, events)

        love.graphics.pop()
    end

    if not game.ended() and not game.turnEnded() then
        DrawEndTurn(game, events)
    end

    DrawStat(game.player1().health, 245, 5)
    DrawStat(game.player2().health, 245, 244)

    love.graphics.print(game.player1().name, 54, 4)
    love.graphics.print(game.player2().name, 54, 243)

    love.graphics.push("all")
    love.graphics.setColor(0xFF, 0xFF, 0xFF,
                           If(game.currentPlayer() == game.player1(), 1, 0.6));
    love.graphics.draw(assets[game.player1().avatar], 2, 2)

    love.graphics.setColor(0xFF, 0xFF, 0xFF,
                           If(game.currentPlayer() == game.player2(), 1, 0.6));
    love.graphics.draw(assets[game.player2().avatar], 2, 242)
    love.graphics.pop()

    love.graphics.pop()
end

function DrawElement(events, element, position, y, value)
    love.graphics.push()
    local x = elementXPosition(position - 1)
    love.graphics.translate(x, y)
    local image = assets["element_" .. element]

    if ClickedInside(0, 0, image:getWidth(), image:getHeight()) then
        table.insert(events, function(game) game.selectElement(element) end)
    end

    love.graphics.draw(image, 0, 0)

    love.graphics.setFont(element_font)
    DrawStat(value, 7, 18, 19)

    -- if y > 200 then
    --     love.graphics.printf(position, 0, 37, image:getWidth(), "center")
    -- end

    love.graphics.pop()
end

function DrawEndTurn(game, events)
    local player1_y = 25
    local player2_y = 265

    local end_turn_x = 440
    local end_turn_y = 0

    if game.currentPlayer() == game.player1() then
        end_turn_y = player1_y + 2
    else
        end_turn_y = player2_y + 2
    end

    love.graphics.draw(assets.end_turn, end_turn_x, end_turn_y)

    if ClickedInside(end_turn_x, end_turn_y, assets.end_turn:getWidth(),
                     assets.end_turn:getHeight()) then
        table.insert(events, function(game) game.endTurn() end)
    end
end

function TestBoardKeys(key, events)
    if BoardShortcutsPositions[key] ~= nil then
        table.insert(events, function(game)
            game.playCard(BoardShortcutsPositions[key])
        end)
    end

    local number = tonumber(key)
    if Elements[number] ~= nil then
        local element = Elements[number]
        table.insert(events, function(game) game.selectElement(element) end)
    end
end
