require "utils"
local stat = require "model/stat"
local effects = require "model.effects"

local waterCards = {}

function waterCards.nixie()
    return {
        name = "Nixie",
        ability = "Causes 200% of damage to fire creatures. Gives owner 1 Water in the beginning of owner's turn.",
        ability_sv = "Orsakar dubbel skada mot Eldvarelser. Ger ägaren 1 Vatten i början av sin tur.",
        spell = "Casting Sea of Sacred increases owner's Water by 1 and reduces Fire by 1.",
        spell_vs = "Sea of Sacred ökar ägarens Vatten med 1 och reducerar Eld med 1.",

        element = "water",
        type = "creature",
        health = stat.health(10),
        attack = stat.new(3),
        level = 4,

        onTurnStart = function(self, game)
            game.owner(self).water.change(1)
        end,
        onAttack = function(self, game)
            local opposingCard = game.opposingCard(self)
            if opposingCard ~= nil and opposingCard.element == "fire" then
                game.damageCard(opposingCard, self, self.attack.value() * 2)
                return false -- no default attack
            end
            return true -- run default attack
        end,
        canCast = function(self, game)
            return game.owner(self).fire.value() >= 1
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            owner.fire.change(-1)
            owner.water.change(1)
        end
    }
end

function waterCards.poseidon()
    return {
        name = "Poseidon",
        ability = "Every time anyone casts Water spell or summons Water creature, opponent suffers 4 damage and owner gains 2 health.",
        ability_sv = "Varje gång någon spelar något Vatten-kort så skadas motståndaren med 4 och ägaren får 2 hälsa.",

        element = "water",
        type = "creature",
        health = stat.health(25),
        attack = stat.new(3),
        level = 8,

        onAnyCardPlayed = function(self, game, playedCard)
            if playedCard.element == "water" and self ~= playedCard then
                local owner = game.owner(self)
                local opponent = game.opponent(owner)
                game.damagePlayer(opponent, self, 4)
                game.healPlayer(owner, 2)
            end
        end
    }
end

function waterCards.iceguard()
    return {
        name = "Ice guard",
        ability = "Reduces all damage done to owner by 50%. Suffers 200% damage from fire.",
        ability_sv = "Minskar all skada mot ägaren med hälften. Tar dubbel skada från Eld.",

        element = "water",
        type = "creature",
        health = stat.health(19),
        attack = stat.new(4),
        level = 5,

        beforePlayerDamage = function(self, game, player, sourceCard, amount)
            return If(player == game.owner(self), math.ceil(amount / 2), amount)
        end,
        onDamage = function(game, self, fromCard, amount)
            if fromCard ~= nil and fromCard.element == "fire" then
                amount = amount * 2
            end
            self.health.change(-amount)
        end
    }
end

function waterCards.icewizard()
    return {
        name = "Ice wizard",
        ability = "Increases Water by 2 every turn. Suffers 200% damage from fire. All damage from Water equal to 1.",
        ability_sv = "Ökar Vatten med 2 varje tur. Tar dubbel skada av Eld. All skada från Vatten lika med 1.",
        spell = "Casting Healing Water heals owner equal to 2 * Water points. Owner loses all Water.",
        spell_sv = "Healing Water läker ägaren med 2 * Vatten, ägaren förlorar allt sitt Vatten.",

        element = "water",
        type = "creature",
        health = stat.health(22),
        attack = stat.new(4),
        level = 10,

        onTurnStart = function(self, game)
            game.owner(self).water.change(2)
        end,
        onDamage = function(game, self, fromCard, amount)
            if fromCard ~= nil and fromCard.element == "fire" then
                amount = amount * 2
            elseif fromCard ~= nil and fromCard.element == "water" then
                amount = 1
            end
            self.health.change(-amount)
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            game.healPlayer(owner, 2 * owner.water.value())
            owner.water.change(-owner.water.value())
        end
    }
end

function waterCards.leviathan()
    return {
        name = "Leviathan",
        ability = "When attacking, each enemy creature suffers 1 damage in addition to standard attack.",
        ability_sv = "När Leviathan attackerar så tar alla fiender 1 skada utöver den vanliga attacken.",
        spell = "Casting Curing heals owner for 4. In exchange, owner loses 1 Water. Cannot cast if owner's Water less than 6.",
        spell_sv = "Curing läker ägaren med 4. Ägaren förlorar 1 Vatten. Går inte att använda om Vatten är mindre än 6.",

        element = "water",
        type = "creature",
        health = stat.health(37),
        attack = stat.new(6),
        level = 11,

        onAttack = function(self, game)
            ForEach(game.enemyCards(self),
                    function(enemyCard)
                game.damageCard(enemyCard, self, 1)
            end)
            return true -- attack normally
        end,
        canCast = function(self, game)
            return game.owner(self).water.value() >= 6
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            game.healPlayer(owner, 4)
            owner.water.change(-1)
        end
    }
end

function waterCards.waterfall()
    return {
        name = "Waterfall",
        ability = "Heals itself for 3 whenever any player casts water spell or summons water creature. Attack equal to owner's Water.",
        ability_sv = "Läker sig själv med 3 när någon spelare castar Vattenkort. Har attack motsvarande ägarens Vatten.",

        element = "water",
        type = "creature",
        health = stat.health(33),
        attack = stat.new(1),
        level = 9,

        onPlay = function(self, game)
            local owner = game.owner(self)
            self.attack = stat.derived(stat.new(0), function()
                return owner.water.value()
            end)
        end,
        onAnyCardPlayed = function(self, game, otherCard)
            if otherCard ~= nil and otherCard.element == "water" then
                game.healCard(self, 3)
            end
        end
    }
end

function waterCards.hydra()
    return {
        name = "Hydra",
        ability = "Attacks oposite and both adjacent slots. Reduces owner's Water by 2 every turn.",
        ability_sv = "Attackerar fiender framför, till höger och vänster. Minskar ägarens Vatten med 2 varje tur.",
        spell = "Consumes friendly unit, receiving up to 50% of its health.",
        spell_sv = "Konsumerar en allierad varelse och läker då halva dess hälsa.",

        element = "water",
        type = "creature",
        health = stat.health(29),
        attack = stat.new(5),
        level = 13,

        onTurnStart = function(self, game)
            game.owner(self).water.change(-2)
        end,
        onAttack = function(self, game)
            local position = game.cardPositionOnBoard(self)
            local enemy = game.opponent(game.owner(self))
            game.damagePosition(enemy, position - 1, self, self.attack.value())
            game.damagePosition(enemy, position + 1, self, self.attack.value())
            return true -- normal attack as well
        end,
        canCast = function(self, game)
            return #Array(game.owner(self).board) > 1 -- self doesn't count
        end,
        onCast = function(self, game)
            game.startSpellTargeting(self, function(card)
                if game.owner(card) == game.owner(self) and card ~= self and
                    card.health ~= nil then
                    game.healCard(self, math.ceil(card.health.value() / 2))
                    game.destroyCard(card)
                    return true
                end
                return false
            end)
        end
    }
end

function waterCards.acidstorm()
    return {
        name = "Acid storm",
        ability = "Each creature suffers up to 16 points of damage. If a player has Poseidon on a field, his creatures left unaffected.",
        ability_sv = "Varje varelse tar 16 skada. Om en spelare har Poseidon i spel så lämnas den spelarens varelser i fred.",
        spell = "Amazingly poisonous magic storm, has no mercy to both friends and foes.",
        spell_sv = "Fräsig magisk storm, skonar varken vänner eller fiender.",

        element = "water",
        type = "spell",
        level = 9,

        onPlay = function(self, game)
            local isPoseidon = function(card)
                return card.name == "Poseidon"
            end
            local acidStormPlayer = function(player)
                if Any(player.board, isPoseidon) then return end
                ForEach(player.board, function(otherCard)
                    game.damageCard(otherCard, self, 16)
                end)
            end
            acidStormPlayer(game.player1())
            acidStormPlayer(game.player2())
        end
    }
end

function waterCards.icebolt()
    return {
        name = "Ice bolt",
        ability = "Inflicts 10 + Water/2 damage to enemy player. Caster suffers 6 damage as a side effect.",
        ability_sv = "Orsakara 10 + Vatten/2 skada till motspelaren. Den som spelar ut tar själv 6 skada.",
        spell = "Large bolt of Ice, fired at a great speed. Superior efficiency!",
        spell_sv = "Pilar av is! cool...",

        element = "water",
        type = "spell",
        level = 7,

        onPlay = function(self, game)
            local owner = game.owner(self)
            local enemy = game.opponent(owner)

            game.damagePlayer(enemy, self,
                              10 + math.floor(owner.water.value() / 2))
            game.damagePlayer(owner, self, 6)
        end
    }
end

function waterCards.paralyze()
    return {
        name = "Paralyze",
        ability = "Paralyzes enemy player and creatures for one turn, so they skip one turn.",
        ability_sv = "Paralyserar motspelaren och dennes varelser så denna står över sin nästa tur.",
        spell = "And only deep silence since then.",
        spell_sv = "Sa du något? Inte det nä...",

        element = "water",
        type = "spell",
        level = 10,

        onPlay = function(self, game)
            local owner = game.owner(self)
            local enemy = game.opponent(owner)

            game.paralyze(enemy)
        end
    }
end

function waterCards.seajustice()
    return {
        name = "Sea justice",
        ability = "Every enemy creature suffers damage equal to its attack -1.",
        ability_sv = "Varje fiendevarelse tar skada motsvarande sin egen attack -1.",
        spell = "Sea has its own, twisted justice. One must get used to it.",
        spell_sv = "Havet har sin egen konstiga idé om rättivsa, bara att svälja.",

        element = "water",
        type = "spell",
        level = 3,

        onPlay = function(self, game)
            ForEach(game.enemyCards(self), function(card)
                if card.attack ~= nil then
                    local amount = math.max(0, card.attack.value() - 1)
                    game.damageCard(card, self, amount)
                end
            end)
        end
    }
end

function waterCards.poison()
    return {
        name = "Poison",
        ability = "Poisons all enemy units so that they lose health every turn, also hits them with 1 damage. Doesn't affect undead.",
        ability_sv = "Förgiftar alla fiendevarelser så de förlorar hälsa varje tur, attackerar dem med 1 dessutom. Påverkar inte Död-varelser.",
        spell = "Taste the poison, feel that this is life!",
        spell_sv = "Smaka på den!",

        element = "water",
        type = "spell",
        level = 3,

        onPlay = function(self, game)
            ForEach(game.enemyCards(self), function(card)
                if card.element ~= "death" then
                    game.damageCard(card, self, 1)
                    effects.add(card, "poison")
                end
            end)
        end
    }
end

return waterCards
