require "utils"

local stat = {}

local function round(value)
    if value < 0 then return math.ceil(value) end
    return math.floor(value)
end

function stat.step(steppable, dt) if steppable ~= nil then steppable.step(dt) end end

function stat.progressions()
    local self = {values = {}}
    local public = {}

    function public.add(progress) table.insert(self.values, progress) end

    function public.addAll(fs, between)
        local delay = 0
        for _, f in pairs(fs) do
            public.add(stat.progress(delay, f))
            delay = delay + between
        end
    end

    function public.step(dt)
        ForEach(self.values, function(steppable) stat.step(steppable, dt) end)
        self.values = Array(Where(self.values, function(progress)
            return not progress.complete()
        end))
    end

    return public
end

function stat.progress(duration, onCompletion)
    local self = {current = 0, to = duration}
    local public = {onCompletion = onCompletion}

    function public.completion() return math.min(self.current / self.to, 1) end

    function public.complete() return public.completion() >= 1 end

    function public.step(dt)
        self.current = self.current + dt
        if public.complete() and public.onCompletion ~= nil then
            public.onCompletion()
        end
    end

    return public
end

function stat.new(value)
    local state = {value = round(value), changes = {}}
    local minValue = 0

    local self = {}

    function self.value() return state.value end

    function self.set(newValue) state.value = round(newValue) end

    function self.canChange(change)
        return state.value + round(change) >= minValue
    end

    function self.change(change)
        change = round(change)
        if change == 0 then return 0 end
        local oldValue = state.value
        local newValue = state.value + change
        if newValue < minValue then newValue = minValue end
        state.value = newValue
        local changeAnimationDurationSeconds = 2
        local newChange = {
            value = newValue - oldValue,
            current = 0,
            to = changeAnimationDurationSeconds,
            completion = 0,
            complete = false
        }
        for i, v in ipairs(state.changes) do
            if v.complete then
                state.changes[i] = newChange
                return oldValue - newValue
            end
        end
        table.insert(state.changes, newChange)
        return oldValue - newValue
    end

    function self.step(dt)
        local allChangesDone = true
        for _, change in pairs(state.changes) do
            change.current = change.current + dt
            change.completion = change.current / change.to
            change.complete = change.completion >= 1
            if not change.complete then allChangesDone = false end
        end

        if allChangesDone then state.changes = {} end
    end

    function self.changes()
        local result = {}
        for key, change in pairs(state.changes) do
            result[key] = {value = change.value, completion = change.completion}
        end
        return result
    end

    return self
end

function stat.derived(baseStat, statFunction)
    local self = baseStat
    local baseValueFunction = self.value

    self.value = function()
        local baseValue = baseValueFunction()
        return statFunction(baseValue)
    end

    return self
end

function stat.health(value)
    value = round(value)
    local self = stat.new(value)
    local maxValue = value

    self.heal = function(amount)
        amount = round(amount)
        if self.value() >= maxValue then return end
        local newValue = self.value() + amount
        if newValue > maxValue then amount = maxValue - self.value() end
        self.change(amount)
    end

    self.gain = function(amount) maxValue = maxValue + amount end

    return self
end

return stat
