require "utils"
local stat = require "model/stat"
local effects = require "model/effects"

local lifeCards = {}

function lifeCards.apostate()
    return {
        name = "Apostate",
        ability = "Steals 2 owner's Life and gives owner 1 Death in the beginning of owner's turn.",
        ability_sv = "Stjäl 2 av ägarens Liv och ger ägaren 1 Död i början av varje tur.",
        spell = "Serves Death. Once cast, Apostate permanently turns into a Banshee. Banshee restores only 1/2 of normal health.",
        spell_sv = "Apostate kan förvandla sig till en Banshee. Banshee får bara hälften av sin normala hälsa.",

        element = "life",
        type = "creature",
        health = stat.health(14),
        attack = stat.new(4),
        level = 5,

        onTurnStart = function(self, game)
            local stolenLife = game.currentPlayer().life.change(-2)
            if self.health ~= nil then
                game.healCard(self, stolenLife)
            end
            game.currentPlayer().death.change(1)
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            local position = game.cardPositionOnBoard(self)
            local banshee = CreateCard("banshee")
            banshee.health.change(-math.floor(banshee.health.value() / 2))
            owner.board[position] = banshee
        end
    }
end

function lifeCards.chimera()
    return {
        name = "Chimera",
        -- I orginal även: Every spell casting costs 50% less for the owner.
        -- Powerful enough as it stands.
        ability = "Whenever you summon creature, you gain health equal to this creature's level.",
        ability_sv = "Närhelst du spelar ut en varlse så får du hälsa motsvarande den varelsens level.",

        element = "life",
        type = "creature",
        health = stat.health(30),
        attack = stat.new(10),
        level = 13,

        onAnyCardPlayed = function(self, game, otherCard)
            if game.owner(self) == game.owner(otherCard) and otherCard.type ==
                "creature" then
                game.healPlayer(game.owner(self), otherCard.level)
            end
        end
    }
end

function lifeCards.magichealer()
    return {
        name = "Magic healer",
        ability = "Whenever owner player loses health, Magic Healer heals player by this amount, losing hit points equally.",
        ability_sv = "När ägaren förlorar hälsa så läker Magic Healer ägaren med samma mängd samtidigt som den förlorar hälsan själv.",

        element = "life",
        type = "creature",
        health = stat.health(12),
        attack = stat.new(2),
        level = 3,

        afterPlayerDamage = function(self, game, damagedPlayer, damagingCard,
                                     amount)
            if game.owner(self) == damagedPlayer then
                -- Remove health without regard to modifiers
                local heal = self.health.change(-amount)
                if self.health.value() <= 0 then
                    game.destroyCard(self)
                end
                game.healPlayer(damagedPlayer, heal)
            end
        end
    }
end

function lifeCards.paladin()
    return {
        name = "Paladin",
        ability = "Brings 300% damage to undead creatures.",
        ability_sv = "Ger 3 gånger så mycket skada mot Dödvarelser.",
        spell = "Casts Exorcizm. Destroys any undead, but suffers 10 damage himself. Owner also loses 2 Life as a cost of this holy casting.",
        spell_sv = "Castar Exorcizm. Förgör alla Dödvarelser, men lider 10 skada själv. Ägaren förlorar också två Liv (inte hälsa) som en kostnad för denna heliga magi.",

        element = "life",
        type = "creature",
        health = stat.health(20),
        attack = stat.new(4),
        level = 8,

        onAttack = function(self, game)
            local opposingCard = game.opposingCard(self)
            if opposingCard ~= nil and opposingCard.element == "death" then
                local damage = self.attack.value() * 3
                game.damageCard(opposingCard, self, damage)
                return false -- attack done
            end
            return true -- normal attack
        end,
        canCast = function(self, game)
            return game.owner(self).life.canChange(-2)
        end,
        onCast = function(self, game)
            local deathCards = Where(game.allCardsOnBoard(), function(card)
                return card.element == "death"
            end)
            ForEach(deathCards,
                    function(card) game.tryToDestroyCard(card, self) end)
            self.health.change(-10)
            if self.health.value() <= 0 then game.destroyCard(self) end
            game.owner(self).life.change(-2)
        end
    }
end

function lifeCards.pegasus()
    return {
        name = "Pegasus",
        ability = "When summoned, each owner's creature is healed for 3. Also, removes harmful spell effects.",
        ability_sv = "När Pegasus spelas ut får alla ägarens varelser +3 hälsa. Tar också bort skadliga spelleffekter.",
        spell = "Holy Strike deals 5 damage to a target creature. If it is undead creature, Pegasus also suffers 3 damage himself. Costs 2 Life.",
        spell_sv = "Holy Strike orsakar 5 skada mot valfri varelse. Om det är en Dödvarelse så tar Pegasus själv 3 skada. Kostar 2 Liv.",

        element = "life",
        type = "creature",
        health = stat.health(15),
        attack = stat.new(6),
        level = 6,

        onPlay = function(self, game)
            ForEach(game.owner(self).board, function(card)
                game.healCard(card, 3)
                effects.removeNegativeEffects(card)
            end)
        end,
        canCast = function(self, game)
            return game.owner(self).life.canChange(-2)
        end,
        onCast = function(self, game)
            game.startSpellTargeting(self, function(targetCard)
                game.damageCard(targetCard, self, 5)
                if targetCard.element == "death" then
                    game.damageCard(self, targetCard, 3)
                end
                game.owner(self).life.change(-2)
            end)
        end
    }
end

function lifeCards.priest()
    return {
        name = "Priest",
        ability = "+2 to Life, -2 to Death elements every turn. Decreases owner's Life by 3 every time owner casts Death spells.",
        ability_sv = "+2 till Liv, -2 Död varje runda. Minskar ägarens Liv med 3 varje gång ägaren kastar en Död-spell.",

        element = "life",
        type = "creature",
        health = stat.health(9),
        attack = stat.new(1),
        level = 4,

        onTurnStart = function(self, game)
            local owner = game.owner(self)
            owner.life.change(2)
            owner.death.change(-2)
        end,
        onAnyCardPlayed = function(self, game, otherCard)
            if game.owner(otherCard) == game.owner(self) and otherCard.element ==
                "death" and otherCard.type == "spell" then
                game.owner(self).life.change(-3)
            end
        end
    }
end

function lifeCards.unicorn()
    return {
        name = "Unicorn",
        ability = "Unicorn reduces damage from spells to owner's creatures by 50%. Cures poison from owner's creatures.",
        ability_sv = "Unicorn minskar skada från spells mot ägarens varelser med hälften. Tar bort poison (gift) från ägarens varelser.",
        spell = "Casts Unicorn Aura. This Aura destroys useful spell effects from enemy creatures. Costs 2 Life.",
        spell_sv = "Castar Unicorn Aura. Tar bort användbara spelleffekter från motståndarvarelser. Kostar 2 Liv.",

        element = "life",
        type = "creature",
        health = stat.health(25),
        attack = stat.new(8),
        level = 9,

        onPlay = function(self, game)
            ForEach(game.owner(self).board,
                    function(card) effects.remove(card, "poison") end)
        end,
        beforeAnyCardDamage = function(self, game, sourceCard, targetCard,
                                       amount)
            if sourceCard ~= nil and game.owner(targetCard) == game.owner(self) and
                (sourceCard.type == "spell" or sourceCard.casting) then
                return math.ceil(amount * 0.5)
            end
            return amount
        end,
        canCast = function(self, game)
            return game.owner(self).life.canChange(-2)
        end,
        onCast = function(self, game)
            ForEach(game.enemyCards(self),
                    function(card)
                effects.removePositiveEffects(card)
            end)
            game.owner(self).life.change(-2)
        end
    }
end

local function castBless(card, game)
    for i, otherCard in pairs(game.currentPlayer().board) do
        if otherCard.element == "death" then
            game.damageCard(otherCard, card, 10)
        elseif otherCard.attack ~= nil then
            otherCard.attack.change(1)
            effects.add(otherCard, "bless")
        end
    end
end

function lifeCards.bless()
    return {
        name = "Bless",
        ability = "All owner's creatures gain +1 to attack, restore 1 point of health with every hit. Undead creatures suffer 10 damage instead.",
        ability_sv = "Alla ägarens varelser får +1 attack och de återställer 1 hälsa med varje träff. Dödvarelser tar 10 skada istället.",
        spell = "Your army's now under God's protection, and your enemy is doomed forever!",
        spell_sv = "Din armé är nu under heligt beskydd och dina fiender är dömda förevigt!",

        element = "life",
        type = "spell",
        level = 3,

        onPlay = castBless
    }
end

local function castPurify(card, game)
    local owner = game.owner(card)
    local opponent = game.opponent(owner)
    local hasLifeCard = Any(game.currentPlayer().board, function(card)
        return card.element == "life" and card.name ~= "Purify"
    end)
    if not hasLifeCard then return end
    for position, opposingCard in pairs(opponent.board) do
        local healCard = owner.board[position]
        if healCard ~= nil and healCard.health ~= nil and opposingCard ~= nil and
            opposingCard.health ~= nil then
            local stolenHealth = math.min(4, opposingCard.health.value())
            game.damageCard(opposingCard, card, 4)
            game.healCard(healCard, stolenHealth)
        end
    end
    game.currentPlayer().health.change(5)
end

function lifeCards.purify()
    return {
        name = "Purify",
        ability = "If owner has Life creatures in play, heals owner for 5 and steals 4 health from each enemy creature, healing opposed creatures.",
        ability_sv = "Om ägaren har Livvarelser i spel, läk ägaren med 5 och stjäl 4 hälsa från varje motståndarvarelser och ge till motstående av dina varelser.",
        spell = "Only pure souls can use God's blessings.",
        spell_sv = "Endast de som är rena i själen kan använda heliga välsignelser.",

        element = "life",
        type = "spell",
        level = 7,

        onPlay = castPurify
    }
end

function lifeCards.godswrath()
    return {
        name = "Gods wrath",
        ability = "All undead on a field are destroyed. Owner receives 3 Life and 10 health for each destroyed creature.",
        ability_sv = "Alla Dödsvarelser i spel förstörs. Ägaren får 3 Liv och 10 hälsa för varje förstörd varelse.",
        spell = "The great day of The Lord is near and coming quickly. That day will be a day of Wrath, a day of distress and anguish.",
        spell_sv = "Den stora dagen för herren är nära, den dagen kommer med ilska.",

        element = "life",
        type = "spell",
        level = 10,

        onPlay = function(self, game)
            ForEach(game.enemyCards(self), function(card)
                if card.element == "death" then
                    local destroyed = game.tryToDestroyCard(card, self)
                    if destroyed then
                        game.owner(self).life.change(3)
                        game.healPlayer(game.owner(self), 10)
                    end
                end
            end)
        end
    }
end

function lifeCards.lifesacrifice()
    return {
        name = "Life sacrifice",
        ability = "Owner loses health equal to his Life. Enemy suffers damage, double of this amount.",
        ability_sv = "Ägaren förlorar hälsa motsvarande sitt värde på Liv. Motståndaren skadas dubbelt detta.",
        spell = "Sacrificing is the true loving act.",
        spell_sv = "Att offra sig är en äkta akt av kärlek.",

        element = "life",
        type = "spell",
        level = 8,

        onPlay = function(self, game)
            local owner = game.owner(self)
            local life = owner.life.value()
            game.damagePlayer(owner, self, life)
            game.damagePlayer(game.opponent(owner), self, 2 * life)
        end
    }
end

function lifeCards.rejuvenation()
    return {
        name = "Rejuvenation",
        ability = "Heals owner equal to his Life Element*3. Owner loses all Life elements. Blessed creatures healed for 3.",
        ability_sv = "Läker ägaren med 3*Liv. Ägaren förlorar hela värdet på Liv. Alla varelser med Bless läker 3.",
        spell = "Now you live again, mortal. Life is the most precious, be careful next time!",
        spell_sv = "Nu lever du igen, du dödliga. Liv är det mest värdefulla, var försiktig nästa gång!",

        element = "life",
        type = "spell",
        level = 6,

        onPlay = function(self, game)
            local owner = game.owner(self)
            local life = owner.life.value()
            game.healPlayer(owner, 3 * life)
            owner.life.change(-owner.life.value())
            ForEach(game.allCardsOnBoard(), function(card)
                if effects.has(card, "bless") then
                    game.healCard(card, 3)
                end
            end)
        end
    }
end

return lifeCards
