local stat = require "model/stat"
require "model/cards"
require "utils"
local stack = require "stack"
local effects = require "model/effects"
local decks = require "model/decks"

local exports = {}

function GenerateDebugPoints() return {20, 20, 20, 20, 20, 20} end

-- Spread 25 points over 6 elements
function GenerateElementPoints()
    local total = 19 -- 25 - 6 initial points
    local elements = {1, 1, 1, 1, 1, 1}
    for i = 1, total do
        local position = 0
        repeat position = math.random(1, #elements) until elements[position] < 5

        elements[position] = elements[position] + 1
    end
    return elements
end

local function createStats(points, player)
    local elements = {"water", "fire", "air", "earth", "life", "death"}
    for i, element in ipairs(elements) do
        player[element] = stat.new(points[i])
    end
end

local function newPlayer(hand, deck)
    local avatar, name = "avatar_life_01", "Player"
    if deck then
        avatar = deck.avatar
        name = deck.name
    end
    local self = {
        health = stat.new(50),
        water = stat.new(14),
        fire = stat.new(14),
        air = stat.new(14),
        earth = stat.new(14),
        life = stat.new(14),
        death = stat.new(14),

        hand = hand,
        avatar = avatar,
        name = name,
        board = {},
        played = false
    }
    createStats(GenerateElementPoints(), self)

    function self.elements()
        return {
            self.water, self.fire, self.air, self.earth, self.life, self.death
        }
    end
    return self
end

local function updatePlayer(dt, player)
    player.health.step(dt)

    player.water.step(dt)
    player.fire.step(dt)
    player.air.step(dt)
    player.earth.step(dt)
    player.life.step(dt)
    player.death.step(dt)

    for _, card in pairs(player.board) do
        stat.step(card.health, dt)
        stat.step(card.attack, dt)
        stat.step(card.dying, dt)
        stat.step(card.attacking, dt)
    end
end

local function cardPosition(inside, card)
    for i, handCard in pairs(inside) do if card == handCard then return i end end
    return nil
end

local function createCards(deck)
    return SortCards(Select(Keys(Where(deck.cards,
                                       function(selected) return selected end)),
                            CreateCard))
end

function exports.new(onEvent, requireSpellSlot, deck1, deck2)
    local allCardsByElement = CardsByElement(Shuffle(AllCards()))
    local hand1, hand2 = DivideCards(allCardsByElement)

    if deck1 and not decks.isEmpty(deck1) then hand1 = createCards(deck1) end
    if deck2 and not decks.isEmpty(deck2) then hand2 = createCards(deck2) end

    local player1 = newPlayer(hand1, deck1)
    local player2 = newPlayer(hand2, deck2)

    local self = {
        player1 = player1,
        player2 = player2,
        currentPlayer = player1,
        turnEnded = false,
        targeting = false,

        loser = nil,

        progressions = stat.progressions(),
        damageStack = stack.new(),

        onEvent = onEvent
    }

    local public = {requireSpellSlot = not requireSpellSlot == false}

    local triggerEvent = function(name, payload)
        if self.onEvent then self.onEvent(name, payload) end
    end

    local function endGame(losingPlayer)
        self.loser = losingPlayer
        public.cancelTargeting()
        triggerEvent("end", {
            winner = public.opponent(losingPlayer),
            loser = losingPlayer
        })
    end

    function public.allCardsOnBoard()
        local result = {}
        for _, v in pairs(self.currentPlayer.board) do
            table.insert(result, v)
        end
        for _, v in pairs(public.opponent(self.currentPlayer).board) do
            table.insert(result, v)
        end
        return result
    end

    function public.currentPlayer() return self.currentPlayer end
    function public.player1() return self.player1 end
    function public.player2() return self.player2 end
    function public.turnEnded() return self.turnEnded end

    function public.update(dt)
        updatePlayer(dt, self.player1)
        updatePlayer(dt, self.player2)
        stat.step(self.progressions, dt)
    end

    function public.damagePlayer(player, fromCard, amount)
        if public.ended() then return end

        ForEach(public.allCardsOnBoard(), function(card)
            if card.beforePlayerDamage then
                amount = card:beforePlayerDamage(public, player, fromCard,
                                                 amount)
            end
        end)

        player.health.change(-amount)
        if player.health.value() <= 0 then
            endGame(player)
        else
            ForEach(public.allCardsOnBoard(), function(card)
                if card.afterPlayerDamage then
                    card:afterPlayerDamage(public, player, fromCard, amount)
                end
            end)
        end
    end

    function public.healPlayer(player, amount)
        if public.ended() then return end
        player.health.change(amount)
    end

    local function damageBlockedBySpellbreaker(targetCard, fromCard)
        return fromCard and effects.has(targetCard, "spellbreaker") and
                   fromCard.type == "spell"
        -- TODO: Protects agains casts? (fromCard.type == "spell" or fromCard.casting)
    end

    function public.damageCard(targetCard, fromCard, amount)
        if public.ended() then return end
        -- Check that this card isn't causing damage to itself to avoid recursion
        if self.damageStack.contains(targetCard) then return end
        if targetCard == nil or effects.has(targetCard, "absolutedefence") or
            damageBlockedBySpellbreaker(targetCard, fromCard) then return end
        if targetCard.health == nil or targetCard.health.value() == 0 then
            return
        end

        self.damageStack.push(targetCard)

        if effects.has(targetCard, "stone") then
            amount = math.max(amount - 2, 0)
        end

        if effects.has(targetCard, "fireshield") and fromCard then
            public.damageCard(fromCard, targetCard, math.floor(amount / 2))
        end

        for _, otherCard in pairs(public.allCardsOnBoard()) do
            if otherCard.beforeAnyCardDamage then
                amount = otherCard.beforeAnyCardDamage(otherCard, public,
                                                       fromCard, targetCard,
                                                       amount)
            end
        end

        if targetCard.onDamage then
            targetCard.onDamage(public, targetCard, fromCard, amount)
        else
            targetCard.health.change(-amount)
        end
        if targetCard.health.value() <= 0 then
            public.destroyCard(targetCard)
        end

        self.damageStack.pop()
    end

    function public.healCard(targetCard, amount)
        if public.ended() then return end
        if targetCard and targetCard.health then
            targetCard.health.heal(amount)
        end
    end

    function public.damagePosition(targetPlayer, position, fromCard, amount)
        if position < 1 or position > 5 then return end
        local targetCard = targetPlayer.board[position]
        if targetCard then
            public.damageCard(targetCard, fromCard, amount)
        else
            public.damagePlayer(targetPlayer, fromCard, amount)
        end
    end

    function public.cardAttack(card, opponent)
        local position = public.cardPositionOnBoard(card)
        if position == nil then
            -- Card died before it could attack
            return
        end
        if card.attack == nil then
            -- A spell probably
            return
        end
        triggerEvent("attack")
        card.attacking = stat.progress(0.6)

        if card.onAttack then
            local shouldContinue = card:onAttack(public)
            if shouldContinue == false then return end
        end
        public.damagePosition(opponent, position, card, card.attack.value())
    end

    function public.changeAttack(card, amount)
        if card.attack then card.attack.change(amount) end
    end

    local function attacks()
        local opponent = public.opponent(self.currentPlayer)
        local cardsInPlay = self.currentPlayer.board
        local attackingCards = Where(cardsInPlay, function(card)
            return card and not card.summonSickness and card.attack ~= nil
        end)

        local generateAttack = function(card)
            return function() public.cardAttack(card, opponent) end
        end

        return Array(Select(attackingCards, generateAttack))
    end

    function public.startSpellTargeting(card, spellFunction)
        self.targeting = true
        self.castingCard = card
        self.targetingWith = spellFunction
        self.currentPlayer.selectedCard = nil
    end

    function public.isTargetingSpell() return self.targeting end

    function public.cancelTargeting()
        self.targeting = false
        self.targetingWith = nil
        if self.castingCard then self.castingCard.casted = false end
        self.castingCard = nil
    end

    function public.opponent(player)
        assert(player, "asked for opponent to no-one")
        if self.player1 == player then
            return self.player2
        else
            return self.player1
        end
    end

    function public.startNewTurn()
        if public.ended() then return end
        if self.currentPlayer.paralyzed then
            -- Skip back to same player that just played after a small wait
            self.progressions.add(stat.progress(2, function()
                public.removeParalyze(self.currentPlayer)
                self.currentPlayer = public.opponent(self.currentPlayer)
                public.startNewTurn()
            end))
            return
        end
        triggerEvent("startturn")
        self.currentPlayer.played = false
        self.turnEnded = false

        for _, card in pairs(self.currentPlayer.board) do
            card.summonSickness = false
            card.casted = false
            effects.remove(card, "absolutedefence")

            if effects.has(card, "fireshield") and card.element ~= "fire" then
                public.damageCard(card, nil, 2)
            end

            if effects.has(card, "bloodlust") then
                public.damageCard(card, nil, 2)
            end

            if effects.has(card, "poison") then
                public.damageCard(card, nil, 2)
            end

            if card.onTurnStart then card:onTurnStart(public) end
        end

        self.currentPlayer.water.change(1)
        self.currentPlayer.fire.change(1)
        self.currentPlayer.air.change(1)
        self.currentPlayer.earth.change(1)
        self.currentPlayer.life.change(1)
        self.currentPlayer.death.change(1)
    end

    function public.endTurn()
        if self.turnEnded or public.ended() then return end

        self.turnEnded = true
        self.currentPlayer.selectedCard = nil
        self.currentPlayer.played = true

        public.cancelTargeting()

        local switchPlayers = function()
            self.currentPlayer = public.opponent(self.currentPlayer)
            public.startNewTurn()
        end

        local endOfTurnActions = attacks()
        table.insert(endOfTurnActions, function()
            ForEach(self.currentPlayer.board,
                    function(card) effects.remove(card, "bloodrage") end)
        end)
        self.progressions.addAll(endOfTurnActions, 0.2)

        local endOfTurnAttacks = attacks()
        local totalAttackTime = If(#endOfTurnAttacks > 0,
                                   0.6 + 0.2 * (#endOfTurnAttacks - 1), 0)
        self.progressions.add(stat.progress(totalAttackTime, switchPlayers))
    end

    function public.selectCard(card)
        local cardIsInPlay = public.cardPositionOnBoard(card) ~= nil
        if self.targeting and cardIsInPlay then
            local castResult = self.targetingWith(card)
            if castResult ~= false then
                triggerEvent("cast")
                self.targeting = false
                self.targetingWith = nil
            end
            return
        end

        if self.currentPlayer.selectedCard == card then
            self.currentPlayer.selectedCard = nil
        else
            self.currentPlayer.selectedCard = card
            triggerEvent("select")
        end
    end

    function public.cardPositionInHand(player, card)
        return cardPosition(player.hand, card)
    end

    function public.cardPositionOnBoard(card)
        local owner = public.owner(card)
        if owner == nil then return nil end
        return cardPosition(owner.board, card)
    end

    function public.cardIsInHand(player, card)
        return public.cardPositionInHand(player, card) ~= nil
    end

    function public.opposingCard(card)
        local owner = public.owner(card)
        local opponent = public.opponent(owner)
        local position = public.cardPositionOnBoard(card)
        return opponent.board[position]
    end

    function public.owner(card)
        if Any(self.player1.board, function(other) return other == card end) then
            return self.player1
        end
        if Any(self.player2.board, function(other) return other == card end) then
            return self.player2
        end
        return nil
    end

    function public.enemyCards(card)
        return Array(Where(public.opponent(public.owner(card)).board, function(
            card) return card and card.type == "creature" end))
    end

    function public.cardIsPlayable(player, card)
        if player.played then return false end

        if not public.cardIsInHand(player, card) then return false end

        local element = self.currentPlayer[card.element]
        return element.canChange(-card.level)
    end

    function public.playCard(position)
        local spellSlotPosition = 6
        if position == spellSlotPosition and public.requireSpellSlot then
            return
        end
        if public.ended() then return end
        if self.currentPlayer.selectedCard == nil then return end
        if self.currentPlayer.board[position] == nil then
            local card = self.currentPlayer.selectedCard
            if card.type ~= "spell" and position == spellSlotPosition then
                return
            end
            if not public.cardIsPlayable(self.currentPlayer, card) then
                return
            end

            local playedCard = CreateCard(card.name)
            self.currentPlayer.board[position] = playedCard
            playedCard.summonSickness = true
            self.currentPlayer.selectedCard = nil
            self.currentPlayer.played = true

            for _, cardInPlay in pairs(public.allCardsOnBoard()) do
                if cardInPlay.onAnyCardPlayed then
                    cardInPlay:onAnyCardPlayed(public, playedCard)
                end
            end

            if playedCard.onPlay then playedCard:onPlay(public) end

            self.currentPlayer[card.element].change(-card.level)

            if playedCard.type == "spell" then
                public.destroyCard(playedCard)
                triggerEvent("spell")
            else
                triggerEvent("summon")
            end
        end
    end

    function public.tryToDestroyCard(targetCard, fromCard)
        if effects.has(targetCard, "spellbreaker") or
            effects.has(targetCard, "absolutedefence") then return false end
        public.destroyCard(targetCard)
        return true
    end

    function public.destroyCard(card)
        if public.ended() then return end
        local owner = public.owner(card)
        local onCompletion = function()
            if card.onCardDies then card:onCardDies(public) end
            local position = public.cardPositionOnBoard(card)
            if position ~= nil then owner.board[position] = nil end
            if self.player1.selectedCard == card then
                self.player1.selectedCard = nil
            end
            if self.player2.selectedCard == card then
                self.player2.selectedCard = nil
            end

            for _, v in pairs(public.allCardsOnBoard()) do
                if v.onAnyOtherCardDies then
                    v:onAnyOtherCardDies(public, card)
                end
            end
        end
        if card.type == "creature" then triggerEvent("dies") end
        card.dying = stat.progress(0.8, onCompletion)
    end

    function public.canCast(card)
        assert(card, "Need a card to check if it can cast")

        if public.ended() or self.targeting then return false end
        local cardOwner = public.owner(card)
        if cardOwner == nil then return false end
        local activePlayer = public.currentPlayer() == cardOwner

        local playerCanCast = activePlayer and not self.turnEnded
        local cardReady = not card.casted and
                              (card.canCast == nil or card:canCast(public))
        local cardCanCast = card.onCast and cardReady

        return playerCanCast and cardCanCast
    end

    function public.casterCard(card) return card.onCast ~= nil end

    function public.strongestCard(cards)
        local highestLevel = 0
        local strongestCard = nil
        for _, card in pairs(cards) do
            if card.level > highestLevel then
                highestLevel = card.level
                strongestCard = card
            end
        end
        return strongestCard
    end

    function public.cast(card)
        if public.canCast(card) then
            card.casting = true
            card:onCast(public)
            if not self.targeting then triggerEvent("cast") end
            card.casting = false
            card.casted = true
        end
    end

    function public.ended() return self.loser ~= nil end

    function public.selectElement(element)
        if self.currentPlayer.selectedElement == element then
            self.currentPlayer.selectedElement = nil
        else
            self.currentPlayer.selectedElement = element
        end
    end

    function public.selectedElement()
        return self.currentPlayer.selectedElement
    end

    function public.filteredCardsInHand()
        if self.currentPlayer.selectedElement == nil then return {} end
        return Array(Where(self.currentPlayer.hand, function(card)
            return card.element == self.currentPlayer.selectedElement
        end))
    end

    function public.paralyze(player)
        ForEach(player.board, function(card)
            effects.add(card, "paralyze")
        end)
        player.paralyzed = true
    end

    function public.removeParalyze(player)
        ForEach(player.board,
                function(card) effects.remove(card, "paralyze") end)
        player.paralyzed = false
    end

    public.effects = effects.effects

    return public
end

return exports
