require "utils"
local stat = require "model/stat"
local effects = require "model/effects"

local fireCards = {}

function fireCards.cerberus()
    return {
        name = "Cerberus",
        ability = "Attacks adjacent enemy units at a half of it's strength.",
        ability_sv = "Attackerar även fiender i rutorna bredvid med hälften av sin styrka.",

        element = "fire",
        type = "creature",
        health = stat.health(8),
        attack = stat.new(4),
        level = 4,

        onAttack = function(self, game)
            local owner = game.owner(self)
            local opponent = game.opponent(owner)
            local position = game.cardPositionOnBoard(self)

            local adjacentCardLeft = opponent.board[position - 1]
            game.damageCard(adjacentCardLeft, self, self.attack.value() / 2)

            local adjacentCardRight = opponent.board[position + 1]
            game.damageCard(adjacentCardRight, self, self.attack.value() / 2)

            return true -- continue with normal attack
        end
    }
end

local function waterCreatures(cards)
    return Array(Where(cards, function(card)
        return
            card ~= nil and card.type == "creature" and card.element == "water"
    end))
end

function fireCards.efreet()
    return {
        name = "Efreet",
        ability = "Those who attacks Efreet, suffers half of damage dealt. On summon deals 6 damage to enemy Water creatures.",
        ability_sv = "De som attackerar Efreet tar själva hälften av skadan de orsakar. Vattenvarelser får 6 i skada när Efreet kommer i spel.",
        spell = "Casts Fire Shield on any owner's creature. Costs 2 Fire. Creatures with Fire Shield returns half the damage received. Fire Shield burns creature from inside, damaging it for 2 points per turn, unless it's a Fire creature.",
        spell_sv = "Castar Fire Shield på valfri av ägarens varelser. Kostar 2 Eld. Fire Shield ger tillbaka hälften av skadan som varelsen tar, men bränner också den inifrån för 2 skada per tur (om det inte är en Eldvarelse).",

        element = "fire",
        type = "creature",
        health = stat.health(33),
        attack = stat.new(6),
        level = 10,

        canCast = function(self, game)
            return game.owner(self).fire.value() >= 2
        end,
        onCast = function(self, game)
            local castFireShield = function(card)
                if game.owner(self) ~= game.owner(card) or
                    effects.has(card, "fireshield") then
                    return false
                end
                game.owner(self).fire.change(-2)
                effects.add(card, "fireshield")
                return true
            end
            game.startSpellTargeting(self, castFireShield)
        end,
        onPlay = function(self, game)
            local enemyWaterCreatues = waterCreatures(game.enemyCards(self))
            ForEach(enemyWaterCreatues,
                    function(creature)
                game.damageCard(creature, self, 6)
            end)
        end,
        onDamage = function(game, self, fromCard, amount)
            game.damageCard(fromCard, fromCard, math.floor(amount / 2))
            self.health.change(-amount)
        end
    }
end

function fireCards.salamander()
    local function adjustOwnersCreaturesAttacks(self, game, amount)
        ForEach(game.owner(self).board,
                function(card) game.changeAttack(card, amount) end)
    end

    return {
        name = "Salamander",
        ability = "Increases attack of all owner's creatures by 2. Increases damage from owner player's spellcastings by 2.",
        ability_sv = "Ökar alla ägarens varelsers attack med 2. Ökar ska som spelarens skada orsakad av spells (även casts) med 2.",

        element = "fire",
        type = "creature",
        health = stat.health(15),
        attack = stat.new(4),
        level = 8,

        onPlay = function(self, game)
            adjustOwnersCreaturesAttacks(self, game, 2)
        end,
        onAnyCardPlayed = function(self, game, otherCard)
            if game.owner(otherCard) == game.owner(self) and otherCard.type ==
                "creature" then game.changeAttack(otherCard, 2) end
        end,
        onCardDies = function(self, game)
            adjustOwnersCreaturesAttacks(self, game, -2)
        end,
        beforePlayerDamage = function(self, game, player, sourceCard, amount)
            if game.owner(sourceCard) == game.owner(self) and
                (sourceCard.type == "spell" or sourceCard.casting) then
                return amount + 2
            end
            return amount
        end,
        beforeAnyCardDamage = function(self, game, sourceCard, targetCard,
                                       amount)
            return self:beforePlayerDamage(game, nil, sourceCard, amount)
        end
    }
end

function fireCards.demon()
    return {
        name = "Demon",
        ability = "Doesn't suffer from Fire and Earth magic (spells and casts).",
        ability_sv = "Tar inte skada av Eld- och Jordmagi (spells och casts).",
        spell = "Whenever Demon casts Fire Breeds owner loses 1 Earth and receives 2 Fire elements.",
        spell_sv = "När Demon castar så förlorar ägaren 1 Jord och får 2 Eld.",

        element = "fire",
        type = "creature",
        health = stat.health(12),
        attack = stat.new(2),
        level = 5,

        onDamage = function(game, self, fromCard, amount)
            if fromCard ~= nil and
                (fromCard.type == "spell" or fromCard.casting) and
                (fromCard.element == "earth" or fromCard.element == "fire") then
                return
            end
            self.health.change(-amount)
        end,
        canCast = function(self, game)
            return game.owner(self).earth.value() >= 1
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            owner.earth.change(-1)
            owner.fire.change(2)
        end
    }
end

function fireCards.vulcan()
    return {
        name = "Vulcan",
        ability = "Immune to harmful Fire spells. When summoned, enemy player loses 3 Fire, and opposed unit suffers 9 damage. Attack equal owner's Fire + 3.",
        ability_sv = "Immun mot skada från Eldspells. Motspelaren förlorar 3 Eld och motstående varelse tar 9 skada när Vulcanen spelas ut. Har en Attack motsvarande ägarens Eld + 3",
        spell = "Casts Volcano Explode. Vulcan dies, but every unit on field suffers damage equal to current Vulcan's health.",
        spell_sv = "Castar Volcano Explode. Vulcan slås ut, men alla varelser i spel tar skada motsvarande Vulcans hälsa.",

        element = "fire",
        type = "creature",
        health = stat.health(27),
        attack = stat.new(0),
        level = 12,

        onPlay = function(self, game)
            game.damageCard(game.opposingCard(self), self, 9)
            game.opponent(game.owner(self)).fire.change(-3)
            self.attack = stat.derived(stat.new(0), function()
                return game.owner(self).fire.value() + 3
            end)
        end,
        onDamage = function(game, self, fromCard, amount)
            if fromCard ~= nil and
                (fromCard.type == "spell" and fromCard.element == "fire") then
                return
            end
            self.health.change(-amount)
        end,
        onCast = function(self, game)
            local damage = self.health.value()
            ForEach(game.allCardsOnBoard(),
                    function(card)
                game.damageCard(card, self, damage)
            end)
            game.destroyCard(self)
        end
    }
end

function fireCards.devil()
    return {
        name = "Devil",
        ability = "Damage from Water is multiplied by 2. Whenever Devil dies, owner suffers 10 damage.",
        ability_sv = "Tar dubbel skada från Vatten. När Devil slås ut så tar ägaren 10 skada.",
        spell = "Sacrificing owner's Fire creature gives 3 Fire to the owner, also healing owner by this amount.",
        spell_sv = "Offrar valfri av ägarens Eldvarelser för 3 Eld och 3 hälsa till ägaren.",

        element = "fire",
        type = "creature",
        health = stat.health(30),
        attack = stat.new(4),
        level = 5,

        onCardDies = function(self, game)
            game.damagePlayer(game.owner(self), self, 10)
        end,
        onDamage = function(game, targetCard, fromCard, amount)
            if fromCard ~= nil and fromCard.element == "water" then
                amount = amount * 2
            end
            targetCard.health.change(-amount)
        end,
        onCast = function(self, game)
            local castSacirifice = function(card)
                if game.owner(self) ~= game.owner(card) or card.element ~=
                    "fire" then return false end
                game.destroyCard(card)
                game.healPlayer(game.owner(self), 3)
                game.owner(self).fire.change(3)
                return true
            end
            game.startSpellTargeting(self, castSacirifice)
        end
    }
end

function fireCards.firelord()
    return {
        name = "Firelord",
        ability = "+1 Fire every turn for both players. Upon dying, brings 8 damage to each player.",
        ability_sv = "+1 Eld varje tur för båda spelarna. När Firelord slås ut tar båda spelarna 8 skada.",
        spell = "Casts BloodLust on any creature. That creature get it's attack increased by 50%, but loses 2 health every turn.",
        spell_sv = "Castar Bloodlust på valfri varelse. Vald varlse får sin attack ökad med 50%, men förlorar 2 hälsa varje tur.",

        element = "fire",
        type = "creature",
        health = stat.health(21),
        attack = stat.new(7),
        level = 11,

        onTurnStart = function(self, game)
            local owner = game.owner(self)
            owner.fire.change(1)
            game.opponent(owner).fire.change(1)
        end,
        onCardDies = function(self, game)
            local owner = game.owner(self)
            game.damagePlayer(owner, self, 8)
            game.damagePlayer(game.opponent(owner), self, 8)
        end,
        onCast = function(self, game)
            game.startSpellTargeting(self, function(card)
                effects.add(card, "bloodlust")
            end)
        end
    }
end

function fireCards.reddrake()
    return {
        name = "Red Drake",
        ability = "When summoned, each enemy suffers 3 damage. Red Drake Suffers no damage from Fire spells and creatures.",
        ability_sv = "Var fiendevarelse tar 3 skada när Red Drake spelas ut. Tar ingen skada från Eldspells eller Eldvarelser.",

        element = "fire",
        type = "creature",
        health = stat.health(16),
        attack = stat.new(5),
        level = 7,

        onPlay = function(self, game)
            ForEach(game.opponent(game.owner(self)).board,
                    function(card) game.damageCard(card, self, 3) end)
        end,
        onDamage = function(game, self, fromCard, amount)
            if fromCard ~= nil and fromCard.element == "fire" then
                return
            end
            self.health.change(-amount)
        end
    }
end

function fireCards.armageddon()
    return {
        name = "Armageddon",
        ability = "All units on a field suffer 25 damage. Each player suffers 10 damage.",
        ability_sv = "Alla varelser på spelplanen tar 25 skada. Varje spelare tar 10 skada.",
        spell = "The ultimate spell of all Orion wizards. The strongest and most harmful. Beware, it's far too powerful!",
        spell_sv = "Den ultimata spellen för alla Orion-magiker. Den starkaste och skadligaste. Var försiktig, den är helt imba.",

        element = "fire",
        type = "spell",
        level = 11,

        onPlay = function(self, game)
            ForEach(game.allCardsOnBoard(),
                    function(card) game.damageCard(card, self, 25) end)
            game.damagePlayer(game.player1(), self, 10)
            game.damagePlayer(game.player2(), self, 10)
        end
    }
end

function fireCards.fireball()
    return {
        name = "Fireball",
        ability = "Each enemy creature suffers damage equal to owner's Fire + 3.",
        ability_sv = "Varje fiendevarelse tar skada motsvarande ägarens Eld + 3.",
        spell = "As easy as it is - a ball of burning fire.",
        spell_sv = "Fireball! Fireball! Fireball!",

        element = "fire",
        type = "spell",
        level = 8,

        onPlay = function(self, game)
            local damage = game.owner(self).fire.value() + 3
            ForEach(game.enemyCards(self),
                    function(card)
                game.damageCard(card, self, damage)
            end)
        end
    }
end

function fireCards.firespikes()
    return {
        name = "Fire Spikes",
        ability = "Deals 3 damage to each enemy creature.",
        ability_sv = "Orsakar 3 skada på varje fiendevarelse.",
        spell = "Cheap and still good. Pure Fire.",
        spell_sv = "Billigt och bra, ren eld.",

        element = "fire",
        type = "spell",
        level = 3,

        onPlay = function(self, game)
            ForEach(game.enemyCards(self),
                    function(card) game.damageCard(card, self, 3) end)
        end
    }
end

function fireCards.flamearrow()
    return {
        name = "Flame Arrow",

        ability = "If enemy has less Fire than owner, enemy suffers damage equal to this difference multiplied by 2. Otherwise 1 damage.",
        ability_sv = "Om motspelaren har mindre Eld än ägaren så tar motspelaren skada motsvarande skillnaden gånger 2. Annars 1 i skada.",
        spell = "Now this is a smart one - a magic arrow made of pure Fire, never misses your foe.",
        spell_sv = "Spela denna om du har mycket mer Eld än din motspelare!",

        element = "fire",
        type = "spell",
        level = 4,

        onPlay = function(self, game)
            local owner = game.owner(self)
            local enemy = game.opponent(owner)
            local damage = If(enemy.fire.value() < owner.fire.value(),
                              (owner.fire.value() - enemy.fire.value()) * 2, 1)
            game.damagePlayer(enemy, self, damage)
        end
    }
end

function fireCards.ritualfire()
    return {
        name = "Ritual Fire",
        ability = "Clears all spell effects from all creatures. Heals Fire creatures for 2, damaging rest for the same.",
        ability_sv = "Tar bort alla spelleffekter från alla varelser. Läker Eldvarelser med 2, skadar övriga med 2.",

        element = "fire",
        type = "spell",
        level = 5,

        onPlay = function(self, game)
            ForEach(game.allCardsOnBoard(), function(card)
                effects.removeEffects(card)
                if card.element == "fire" then
                    game.healCard(card, 2)
                else
                    game.damageCard(card, self, 2)
                end
            end)
        end
    }
end

return fireCards
