local effects = {}
local stat = require "model/stat"

local all = {}

local function bless(card)
    if card.hasBlessLogic then return end
    card.hasBlessLogic = true

    local baseOnAttack = card.onAttack
    card.onAttack = function(_, game)
        -- Check if still blessed to avoid un-overriding the function if losing bless
        if effects.has(card, "bless") then game.healCard(card, 1) end
        if baseOnAttack ~= nil then
            return baseOnAttack(card, game)
        else
            return true -- continue with default attack
        end
    end
end

local function bloodlust(card)
    if card == nil then return false end
    if card.attack and not card.hasBloodlustLogic then
        card.hasBloodlustLogic = true
        card.attack = stat.derived(card.attack, function(baseAttack)
            return If(effects.has(card, "bloodlust"),
                      math.floor(baseAttack * 1.5), baseAttack)
        end)
    end
end

local function bloodrage(card)
    if card == nil then return false end
    if card.attack and not card.hasBloodrageLogic then
        card.hasBloodrageLogic = true
        card.attack = stat.derived(card.attack, function(baseAttack)
            return
                If(effects.has(card, "bloodrage"), baseAttack * 2, baseAttack)
        end)
    end
end

local function weakness(card)
    if card == nil then return false end
    if card.attack and not card.hasWeaknessLogic then
        card.hasWeaknessLogic = true
        card.attack = stat.derived(card.attack, function(baseAttack)
            return If(effects.has(card, "weakness"),
                      math.floor(baseAttack * 0.5), baseAttack)
        end)
    end
end

all.bless = {name = "bless", positive = true, onAdd = bless}
all.spellbreaker = {name = "spellbreaker", positive = true}
all.stone = {name = "stone", positive = true}
all.absolutedefence = {name = "absolutedefence", positive = true}
all.fireshield = {name = "fireshield", positive = true}
all.bloodlust = {name = "bloodlust", positive = true, onAdd = bloodlust}
all.bloodrage = {name = "bloodrage", positive = true, onAdd = bloodrage}
all.poison = {name = "poison", positive = false}
all.paralyze = {name = "paralyze", positive = false}
all.weakness = {name = "weakness", positive = false, onAdd = weakness}

function effects.has(card, effect)
    if card.effects == nil then return false end
    return card.effects[effect] == true
end

function effects.add(card, name)
    local effect = all[name]
    assert(effect, "Unknown effect " .. name)
    if effects.has(card, "spellbreaker") or effects.has(card, "absolutedefence") and
        not effect.positive then return end
    if card.effects == nil then card.effects = {} end

    if effects.has(card, effect) then return end
    card.effects[name] = true

    if effect.onAdd ~= nil then effect.onAdd(card) end
end

function effects.remove(card, effect)
    if card == nil then return end
    if card.effects ~= nil then card.effects[effect] = false end
end

function effects.removeEffects(card)
    if effects.has(card, "spellbreaker") then return end

    card.effects = {}
    card.magicLink = nil
end

function effects.removeNegativeEffects(card)
    ForEach(effects.effects(card), function(effect)
        if not all[effect].positive then effects.remove(card, effect) end
    end)
    card.magicLink = nil
end

function effects.removePositiveEffects(card)
    ForEach(effects.effects(card), function(effect)
        if all[effect].positive then effects.remove(card, effect) end
    end)
end

function effects.effects(card)
    if card.effects == nil then return {} end
    return Keys(Where(card.effects, function(applied) return applied end))
end

function effects.stealEffects(fromCard, toCard)
    if effects.has(fromCard, "absolutedefence") then return end

    ForEach(effects.effects(fromCard), function(effect)
        effects.remove(fromCard, effect)
        effects.add(toCard, effect)
    end)
end

return effects
