require "utils"
local stat = require "model/stat"
local effects = require "model/effects"

local earthCards = {}

function earthCards.satyr()
    return {
        name = "Satyr",
        ability = "Increases Earth by 1 every turn.",
        ability_sv = "Ökar Jord med 1 varje tur.",
        spell = "Once Satyr casts Dissolve, it dies and creature in the opposed slot suffers 5 damage. If there's no creature, damage dealt to enemy player.",
        spell_sv = "När Satyr castar Dissolve så dör den och varelsen på motstående plats får 5 skada. Om där inte finns någon varelse så tar motspelaren skadan.",

        element = "earth",
        type = "creature",
        health = stat.health(10),
        attack = stat.new(3),
        level = 2,

        onTurnStart = function(self, game)
            game.owner(self).earth.change(1)
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            game.damagePosition(game.opponent(owner),
                                game.cardPositionOnBoard(self), self, 5)
            game.destroyCard(self)
        end
    }
end

function earthCards.golem()
    return {
        name = "Golem",
        ability = "Regenerates 3 health every turn. While owner's Earth less than 3, it suffers 3 damage instead.",
        ability_sv = "Återhämtar sig med 3 hälsa varje tur. Om ägarens Jord är mindre än 3 så tar den istället 3 skada.",

        element = "earth",
        type = "creature",
        health = stat.health(15),
        attack = stat.new(4),
        level = 5,

        onTurnStart = function(self, game)
            if game.currentPlayer().earth.value() < 3 then
                game.damageCard(self, nil, 3)
            else
                game.healCard(self, 3)
            end
        end
    }
end

function earthCards.centaur()
    return {
        name = "Centaur",
        ability = "Attacks the same turn he was summoned (No summon sickness).",
        ability_sv = "Attackerar samma tur som han blir frammanad (ingen summon sickness).",
        spell = "Strikes magic arrow into enemy player, dealing 3 damage. Costs 1 Earth.",
        spell_sv = "Skjuter en magisk pil på motspelaren för 3 skada. Kostar 1 Jord.",

        element = "earth",
        type = "creature",
        health = stat.health(14),
        attack = stat.new(5),
        level = 6,

        onPlay = function(self) self.summonSickness = false end,
        canCast = function(self, game)
            local x = game.owner(self)
            return x.earth.value() >= 1
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            local opponent = game.opponent(owner)
            game.damagePlayer(opponent, self, 3)
            owner.earth.change(-1)
        end
    }
end

function earthCards.dryad()
    local adjustAttack = function(game, affectedCard, multiplier)
        if affectedCard == nil then return end
        if affectedCard.element == "earth" then
            game.changeAttack(affectedCard, 3 * multiplier)
        else
            game.changeAttack(affectedCard, 2 * multiplier)

        end
    end

    local function adjustAdjacentAttacks(game, card, multiplier)
        local owner = game.owner(card)
        local position = game.cardPositionOnBoard(card)
        local leftCard = owner.board[position - 1]
        local rightCard = owner.board[position + 1]

        adjustAttack(game, leftCard, multiplier)
        adjustAttack(game, rightCard, multiplier)
    end

    local function isAdjacent(game, cardA, cardB)
        local positionA = game.cardPositionOnBoard(cardA)
        local positionB = game.cardPositionOnBoard(cardB)
        return positionA == positionB - 1 or positionA == positionB + 1
    end

    return {
        name = "Dryad",
        ability = "Adjacent owner creatures' attack increases by 2, and if it's Earth creature, by 3. Dryad regenerates 2 whenever Earth spell cast.",
        ability_sv = "Varelser bredvid får +2 på attack, +3 om det är en Jord-varelse. Dryad återhämtar 2 hälsa när en Jord-magi spelas ut.",

        element = "earth",
        type = "creature",
        health = stat.health(12),
        attack = stat.new(4),
        level = 4,

        onCardDies = function(self, game)
            adjustAdjacentAttacks(game, self, -1)
        end,
        onAnyCardPlayed = function(self, game, otherCard)
            if otherCard.type == "spell" and otherCard.element == "earth" then
                game.healCard(self, 2)
            end
            local owner = game.owner(self)
            local playingPlayer = game.owner(otherCard)
            if otherCard.type == "creature" and owner == playingPlayer and
                isAdjacent(game, self, otherCard) then
                adjustAttack(game, otherCard, 1)
            end
        end,
        onPlay = function(self, game)
            adjustAdjacentAttacks(game, self, 1)
        end
    }
end

function earthCards.echidna()
    return {
        name = "Echidna",
        ability = "When attacks, poisons her target for 2. In the beginning of owner's turn, Echidna hits all poisoned creatures for 1.",
        ability_sv = "När Echidna attackerar så förgiftar hon fienden med 2 skada. I början av ägarens tur så attackerar Echidna alla förgiftade monster för 1 skada.",

        element = "earth",
        type = "creature",
        health = stat.health(26),
        attack = stat.new(7),
        level = 10,

        onTurnStart = function(self, game)
            local poisonedCards = Where(game.allCardsOnBoard(), function(card)
                return effects.has(card, "poison")
            end)
            ForEach(poisonedCards,
                    function(card) game.damageCard(card, self, 1) end)
        end,
        onAttack = function(self, game)
            local target = game.opposingCard(self)
            if target ~= nil then
                effects.add(target, "poison")
                game.damageCard(target, self, 2)
            end
            return true
        end
    }
end

function earthCards.elemental()
    return {
        name = "Elemental",
        ability = "No damage from creatures with level less than 4.",
        ability_sv = "Tar ingen skada från varelser med level mindre än 4.",
        spell = "Magic Link: links self to enemy creature. That creature suffers same damage as Elemental does. Cost 1 of each element.",
        spell_sv = "Magisk länk: kopplar sig till en fiendevarelse. Den varelsen tar samma skada som Elemental tar. Kostar 1 av varje element.",

        element = "earth",
        type = "creature",
        health = stat.health(26),
        attack = stat.new(7),
        level = 10,

        onDamage = function(game, targetCard, fromCard, amount)
            if fromCard ~= nil and
                (fromCard.level < 4 and fromCard.type == "creature") then
                return
            end

            targetCard.health.change(-amount)

            if targetCard.magicLink ~= nil then
                game.damageCard(targetCard.magicLink, fromCard, amount)
            end
        end,
        canCast = function(self, game)
            local owner = game.owner(self)
            return not Any(owner.elements(),
                           function(element)
                return element.value() < 1
            end)
        end,
        onCast = function(self, game)
            game.startSpellTargeting(self, function(targetCard)
                if game.owner(self) == game.owner(targetCard) then
                    return false
                end
                local owner = game.owner(self)
                ForEach(owner.elements(),
                        function(element) element.change(-1) end)
                self.magicLink = targetCard
                return true
            end)
        end
    }
end

function earthCards.ent()
    return {
        name = "Ent",
        ability = "Attacks opposed unit and enemy player at the same time.",
        ability_sv = "Attackerar motstående varelse och motspelaren på samma gång.",
        spell = "Casts Entangle Roots, damaging each enemy for 1 and losing 2 points of own health.",
        spell_sv = "Castar Entangle Roots, skadar varje fiende med 1 och förlorar själv 2 skada.",

        element = "earth",
        type = "creature",
        health = stat.health(22),
        attack = stat.new(3),
        level = 7,

        onAttack = function(self, game)
            local owner = game.owner(self)
            local opponent = game.opponent(owner)
            game.damagePlayer(opponent, self, self.attack.value())
            local opposingCard = game.opposingCard(self)
            game.damageCard(opposingCard, self, self.attack.value())
            return false -- no default attack
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            local opponent = game.opponent(owner)
            ForEach(opponent.board,
                    function(card) game.damageCard(card, self, 1) end)
            game.damageCard(self, self, 2)
        end
    }
end

function earthCards.forestspirit()
    return {
        name = "Forest Spirit",
        ability = "Damage from all non-magical attacks and abilities equal to 1.",
        ability_sv = "Skada från alla icke-magiska attacker och färdigheter lika med 1.",
        spell = "Casts Youth of Forest, increasing owner player's health by 5. Costs two Earth elements.",
        spell_sv = "Castar Youth of Forest, ökar ägarens hälsa med 5. Kostar 2 Jord.",

        element = "earth",
        type = "creature",
        health = stat.health(3),
        attack = stat.new(2),
        level = 3,

        onDamage = function(game, targetCard, fromCard, amount)
            if fromCard ~= nil and fromCard.type == "creature" and
                not fromCard.casting then amount = 1 end
            targetCard.health.change(-amount)
        end,
        canCast = function(self, game)
            return game.owner(self).earth.value() > 2
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            owner.health.change(5)
            owner.earth.change(-2)
        end
    }
end

function earthCards.earthquake()
    return {
        name = "Earthquake",
        ability = "Hits each creature for 15 damage. Doesn't affect owner's creatures, if onwer's Earth > 12.",
        ability_sv = "Skadar varje varelse med 15. Påverkar inte ägarens varelser, om dennes Jord är större än 12.",
        spell = "Even the earth itself is a powerful weapon.",
        spell_sv = "Till och med jorden själv är ett kraftfullt vapen.",

        element = "earth",
        type = "spell",
        level = 10,

        onPlay = function(self, game)
            local damageCards = function(cards)
                for _, card in pairs(cards) do
                    if self ~= card then
                        game.damageCard(card, self, 15)
                    end
                end
            end
            local owner = game.owner(self)
            if owner.earth.value() <= 12 then
                damageCards(owner.board)
            end
            damageCards(game.opponent(owner).board)
        end
    }
end

function earthCards.revival()
    return {
        name = "Revival",
        ability = "Heals all friendly creatures for 4. Gives owner 2 health for each of his creatures on a field.",
        ability_sv = "Läker alla vänliga varelser med 4. Ger ägaren 2 hälsa för varje av dennes varelse på brädet.",
        spell = "Heal me! Heal me!",
        spell_sv = "Läk mig! Läk mig!",

        element = "earth",
        type = "spell",
        level = 5,

        onPlay = function(self, game)
            local owner = game.owner(self)
            local heal = 0
            for _, card in pairs(owner.board) do
                if card ~= self and card.health ~= nil then
                    game.healCard(card, 4)
                    heal = heal + 2
                end
            end
            owner.health.change(heal)
        end
    }
end

function earthCards.absolutedefence()
    return {
        name = "Absolute Defence",
        ability = "Owner's creatures gain protection from all attacks. This only lasts one turn until next owner's turn.",
        ability_sv = "Ägarens varelser får skydd från alla attacker. Detta varar bara till ägarens nästa tur.",
        spell = "It's just like an unpenetrable wall has suddenly appeared. Anyone under your command will survive anything!",
        spell_sv = "Det är som en ogenomtränglig vägg som plötsligt dyker upp. Alla dina varelser överlever vad som helst!",

        element = "earth",
        type = "spell",
        level = 7,

        onPlay = function(self, game)
            local ownCards = ExceptValue(game.owner(self).board, self)
            local defend = function(card)
                effects.add(card, "absolutedefence")
            end
            ForEach(ownCards, defend)
        end
    }
end

function earthCards.quicksand()
    return {
        name = "Quicksand",
        ability = "Kills all enemy creatures of level less than 5.",
        ability_sv = "Dödar alla fiendevarelser med mindre än 5 i level",
        spell = "Only the skilled one can survive the swamp's most dangerous weapon.",
        spell_sv = "Bara de med stor färdighet kan överleva träskets farligaste vapen.",

        element = "earth",
        type = "spell",
        level = 6,

        onPlay = function(self, game)
            ForEach(game.enemyCards(self), function(enemyCard)
                if enemyCard.level < 5 then
                    game.tryToDestroyCard(enemyCard, self)
                end
            end)
        end
    }
end

function earthCards.restructure()
    return {
        name = "Restructure",
        ability = "All owner's creatures gain +3 health to their maximum, healing for 6 in the same time.",
        ability_sv = "Alla ägarens varelser får +3 max hälsa, läker också 6 samtidigt.",
        spell = "Scatter to pieces, connect once again. Now you are stronger, none shall remain!",
        spell_sv = "Dela upp i delar, bygg ihop igen. Nu är du starkare, ingen skall förbli!",

        element = "earth",
        type = "spell",
        level = 6,

        onPlay = function(self, game)
            -- TODO: Handle as effect, is it removable?
            local ownCards = ExceptValue(game.owner(self).board, self)
            ForEach(ownCards, function(card)
                if card.health ~= nil then
                    card.health.gain(3)
                    card.health.heal(6)
                end
            end)
        end
    }
end

return earthCards
