require "utils"
local stat = require "model/stat"
local effects = require "model/effects"

local deathCards = {}

function deathCards.banshee()
    return {
        name = "Banshee",
        ability = "When summoned, deals 8 damage to enemy. When attacks enemy player, dies and enemy player suffers 10 points of extra damage.",
        ability_sv = "När Banshee spelas ut får motspelaren 8 skada. Om Banshee attackerar motspelaren så dör Banshee och orsakar 10 extra skada.",

        element = "death",
        type = "creature",
        health = stat.health(12),
        attack = stat.new(5),
        level = 7,

        onPlay = function(self, game)
            game.damagePlayer(game.opponent(game.owner(self)), self, 8)
        end,
        onAttack = function(self, game)
            if game.opposingCard(self) == nil then
                local owner = game.owner(self)
                local opponent = game.opponent(owner)
                game.damagePlayer(opponent, self, 10)
                game.destroyCard(self)
                -- continue with normal attack
            end
            return true -- do normal attack
        end
    }
end

function deathCards.darklord()
    return {
        name = "Dark lord",
        ability = "Whenever creature dies, Darklord heals owner for 3 and regenerates self for 2.",
        ability_sv = "Varje gång någon varelse dör så helar Darklord ägaren med 3 hälsa och sig själv med 2.",
        spell = "Steal Spell steals all spell effects from any enemy creature, DarkLord receives these enchants. Owner loses 1 Death.",
        spell_sv = "Steal Spell stjäler alla spelleffekter från en fiendevarelse, Darklord får dessa effekter. Ägaren förlorar 1 Död.",

        element = "death",
        type = "creature",
        health = stat.health(14),
        attack = stat.new(4),
        level = 8,

        onAnyOtherCardDies = function(self, game, otherCard)
            if otherCard.type == "creature" then
                game.healPlayer(game.owner(self), 3)
                game.healCard(self, 2)
            end
        end,
        onCast = function(self, game)
            game.startSpellTargeting(self, function(targetCard)
                if game.owner(targetCard) ~= game.owner(self) then
                    effects.stealEffects(targetCard, self)
                    game.owner(self).death.change(-1)
                    return true
                end
                return false
            end)
        end
    }
end

function deathCards.ghost()
    return {
        name = "Ghost",
        ability = "Splits damage he recieves between owner and self. When suffers from spell, Ghost recieves 200% of normal damage.",
        ability_sv = "Delar skada den tar mellan sig själv och ägaren. När Ghost tar skada från en spell så blir skadan dubbel.",
        spell = "Casts Blood Ritual. As a result, owner loses 4 health, but receives one Death.",
        spell_sv = "Castar Blood Ritual. Ägaren förlorar 4 hälsa, men får 1 Död.",

        element = "death",
        type = "creature",
        health = stat.health(13),
        attack = stat.new(3),
        level = 3,

        onDamage = function(game, targetCard, fromCard, amount)
            local spell = fromCard and
                              (fromCard.type == "spell" or fromCard.casting)
            if spell then amount = 2 * amount end
            local cardDamage = math.ceil(amount / 2)
            local playerDamage = math.floor(amount / 2)

            targetCard.health.change(-cardDamage)
            game.damagePlayer(game.owner(targetCard), fromCard, playerDamage)
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            game.damagePlayer(owner, self, 4)
            owner.death.change(1)
        end
    }
end

function deathCards.grimreaper()
    return {
        name = "Grim reaper",
        ability = "Whenever creature dies, increases owner's Death by one.",
        ability_sv = "När en varelse dör så ökas ägarens Död med 1.",
        spell = "Consumes target enemy creature of level 3 or less. Owner player loses 3 Death elements.",
        spell_sv = "Konsumerar valfri fiendevarelse med level 3 eller mindre. Ägaren förlorar 3 Död.",

        element = "death",
        type = "creature",
        health = stat.health(25),
        attack = stat.new(7),
        level = 11,

        onAnyOtherCardDies = function(self, game, otherCard)
            if otherCard.type == "creature" then
                game.owner(self).death.change(1)
            end
        end,
        canCast = function(self, game)
            return game.owner(self).death.canChange(-3)
        end,
        onCast = function(self, game)
            game.startSpellTargeting(self, function(card)
                if card.type == "creature" and card.level <= 3 then
                    game.tryToDestroyCard(card, self)
                    game.owner(self).death.change(-3)
                    return true
                end
                return false
            end)
        end
    }
end

function deathCards.lich()
    return {
        name = "Lich",
        ability = "When summoned, deals 10 damage to creature in the opposite slot and two adjacent slots. Extra 5 damage to Life units.",
        ability_sv = "När Lich spelas så får motstående varelse och de bredvid 10 skada. Liv-varelser får 5 extra skada.",
        spell = "Casts Death Bolt, hitting enemy player with 7 of damage. Owner loses 5 Death. If owner's Death becomes zero, he suffers 10 damage.",
        spell_sv = "Castar Death Bolt, träffar motspelaren med 7 skada. Ägaren förlorar 5 Död. Om ägarens Död blir 0 så tar ägaren 10 skada.",

        element = "death",
        type = "creature",
        health = stat.health(18),
        attack = stat.new(7),
        level = 10,

        onPlay = function(self, game)
            local position = game.cardPositionOnBoard(self)
            local opponent = game.opponent(game.owner(self))
            local doDamage = function(card)
                if card == nil then return end
                local amount = If(card.element == "life", 15, 10)
                game.damageCard(card, self, amount)
            end
            doDamage(opponent.board[position - 1])
            doDamage(opponent.board[position])
            doDamage(opponent.board[position + 1])
        end,
        onCast = function(self, game)
            local owner = game.owner(self)
            local opponent = game.opponent(owner)
            game.damagePlayer(opponent, self, 7)
            owner.death.change(-5)
            if owner.death.value() == 0 then
                game.damagePlayer(owner, self, 10)
            end
        end
    }
end

function deathCards.vampire()
    return {
        name = "Vampire",
        ability = "When attacks living creature, restores health equal to 50% of damage dealt. Maximum 30 health.",
        ability_sv = "När Vampire attackerar en levande varelse så läker den sig själv motsvarande halva skada. Vampire kan ha max 30 hälsa.",

        element = "death",
        type = "creature",
        health = stat.health(22),
        attack = stat.new(6),
        level = 9,

        onPlay = function(self, game)
            self.health.gain(8) -- Increase max health to 30
        end,
        onAttack = function(self, game)
            local opposingCard = game.opposingCard(self)
            if opposingCard and opposingCard.type == "creature" and
                opposingCard.element ~= "death" then
                local damage = self.attack.value()
                local healthBefore = opposingCard.health.value()
                game.damageCard(opposingCard, self, damage)
                local healthAfter = opposingCard.health.value()
                local dealt = math.max(healthBefore - healthAfter, 0)
                local restore = math.floor(dealt / 2)
                game.healCard(self, restore)
                return false -- attack done
            end
        end
    }
end

function deathCards.werewolf()
    return {
        name = "Werewolf",
        ability = "When dies, becomes a ghost.",
        ability_sv = "Werewolf blir en Ghost när den slås ut.",
        spell = "Casts Blood Rage on self. Strikes twice as hard this turn, but owner loses 3 Death points on casting.",
        spell_sv = "Castar Blood Rage på sig själv. Slår dubbelt så hårt denna tur, men ägaren förlorar 3 Död.",

        element = "death",
        type = "creature",
        health = stat.health(16),
        attack = stat.new(6),
        level = 6,

        onCardDies = function(self, game)
            local position = game.cardPositionOnBoard(self)
            game.owner(self).board[position] = CreateCard("ghost")
        end,
        canCast = function(self, game)
            return game.owner(self).death.value() >= 3
        end,
        onCast = function(self, game)
            effects.add(self, "bloodrage")
            game.owner(self).death.change(-3)
        end
    }
end

function deathCards.curse()
    return {
        name = "Curse",
        ability = "Reduces all enemy elements by 1.",
        ability_sv = "Minskar motspelarens alla element med 1",
        spell = "Curse and Doom are now your enemy's only guests.",
        spell_sv = "Anfäkta och anamma är nu din fiendes enda gäster.",

        element = "death",
        type = "spell",
        level = 4,

        onPlay = function(self, game)
            local owner = game.owner(self)
            local opponent = game.opponent(owner)
            ForEach(opponent.elements(),
                    function(element) element.change(-1) end)
        end
    }
end

function deathCards.chaosvortex()
    return {
        name = "Chaos vortex",
        ability = "Banishes each creature into hell. Each banished creature gives caster 1 Death.",
        ability_sv = "Skickar alla varelser till helvetet. Ägaren får 1 Död per varelse.",
        spell = "Whenever one unfolds Chaos, no mortal can stand its fearful ugly nature.",
        spell_sv = "När det blir kaos finns det ingen som klarar sig.",

        element = "death",
        type = "spell",
        level = 13,

        onPlay = function(self, game)
            local owner = game.owner(self)
            for _, card in pairs(game.allCardsOnBoard()) do
                if card and card.type == "creature" then
                    local destroyed = game.tryToDestroyCard(card, self)
                    if destroyed then owner.death.change(1) end
                end
            end
        end
    }
end

function deathCards.coverofdarkness()
    return {
        name = "Cover of darkness",
        ability = "All living creatures suffer 13 damage. All undead creatures healed for 5.",
        ability_sv = "Alla levande varelser får 13 i skada. All odöda varelser läker med 5.",
        spell = "The Lord of Chaos most useful tool. Your army of darkness shall reign forever.",
        spell_sv = "Kaosets mästares mest användbara verktyg, din armé av mörker kommer härska förevigt.",

        element = "death",
        type = "spell",
        level = 11,

        onPlay = function(self, game)
            for _, card in pairs(game.allCardsOnBoard()) do
                if card and card.element ~= "death" then
                    game.damageCard(card, self, 13)
                else
                    game.healCard(card, 5)
                end
            end
        end
    }
end

function deathCards.steallife()
    return {
        name = "Steal life",
        ability = "If owner's Death less than 8, steals 5 health from enemy player. Otherwise steals Death + 5.",
        ability_sv = "Om ägarens Död är mindre än 8, stjäl 5 hälsa från motspelaren. Annars stjäl Död + 5.",
        spell = "Death's cold vampiric touch. So painful and surreal..",
        spell_sv = "Dödens kalla hand, så smärtsam, så surrealistisk.",

        element = "death",
        type = "spell",
        level = 6,

        onPlay = function(self, game)
            local owner = game.owner(self)
            local steal =
                If(owner.death.value() < 8, 5, owner.death.value() + 5)

            game.damagePlayer(game.opponent(owner), self, steal)
            game.healPlayer(owner, steal)
        end
    }
end

function deathCards.totalweakness()
    return {
        name = "Total weakness",
        ability = "Every enemy creature suffers effect of Weakness: its attack decreased by 50% (rounded down).",
        ability_sv = "Alla fiendevarelser får effekten Weakness: deras attack minskas till hälften (avrundat nedåt).",
        spell = "Make the strongest the weakest, and then assasinate him.",
        spell_sv = "Gör den starka till din svaga.",

        element = "death",
        type = "spell",
        level = 8,

        onPlay = function(self, game)
            for _, card in pairs(game.enemyCards(self)) do
                effects.add(card, "weakness")
            end
        end
    }
end

return deathCards
