require "utils"
local stat = require "model/stat"
local effects = require "model/effects"

local airCards = {}

function airCards.fairy()
    return {
        name = "Fairy",
        ability = "Increases its attack by 1 for each creature, killed on a field.",
        ability_sv = "Ökar sin attack med 1 för varje varelse som slås ut från spelplanen",
        spell = "Enslave Mind forces the highest level enemy creature to attack it's owner. Costs 1 Air.",
        spell_sv = "Enslave Mind tvinger den fiende som har högst level att attacker sin ägare. Kostar 1 Luft.",

        element = "air",
        type = "creature",
        health = stat.health(7),
        attack = stat.new(3),
        level = 3,

        onAnyOtherCardDies = function(self, game, otherCard)
            if otherCard.type == "creature" then
                self.attack.change(1)
            end
        end,
        canCast = function(self, game)
            return game.currentPlayer().air.value() >= 1
        end,
        onCast = function(self, game)
            local opponent = game.opponent(game.owner(self))
            local strongestEnemyCard = game.strongestCard(game.enemyCards(self))
            if strongestEnemyCard ~= nil then
                game.damagePlayer(opponent, strongestEnemyCard,
                                  strongestEnemyCard.attack.value())
            end
            game.currentPlayer().air.change(-1)
        end
    }
end

function airCards.gargoyle()
    return {
        name = "Gargoyle",
        ability = "Suffers no damage from Earth and Air spells.",
        ability_sv = "Tar ingen skada från Jord- eller Luftspells.",
        spell = "Casts STONE on self, as effect turns to stone. In stone form Gargoyle reduces damage done to it by 2. Owner loses 3 Air and 1 Earth.",
        spell_sv = "Kastar Stone på sig själv, vilket förvandlar den till en sten. I stenform så minskas skadad den tar med 2. Ägaren förlorar 3 Luft och 1 Jord.",

        element = "air",
        type = "creature",
        health = stat.health(16),
        attack = stat.new(4),
        level = 5,

        -- TODO: Needs the point to allow cast?
        canCast = function(self, game)
            return game.currentPlayer().air.value() >= 3 and
                       game.currentPlayer().earth.value() >= 1 and
                       not effects.has(self, "stone")
        end,
        onCast = function(self, game)
            effects.add(self, "stone")
            game.currentPlayer().air.change(-3)
            game.currentPlayer().earth.change(-1)
        end,
        onDamage = function(game, targetCard, fromCard, amount)
            if fromCard ~= nil and
                (fromCard.type == "spell" or fromCard.casting) and
                (fromCard.element == "earth" or fromCard.element == "air") then
                return
            end
            targetCard.health.change(-amount)
        end
    }
end

function airCards.manticore()
    return {
        name = "Manticore",
        ability = "Attacks casters with additional 3 damage. Only suffers 50% damage from spells.",
        ability_sv = "Attackerar casters med 3 extra skada. Tar bara halva skadan från spells.",
        spell = "Casts Memory Loss. Target enemy creature permanently loses ability to cast.",
        spell_sv = "Castar Memory Loss. Valfri fiende tappar minnet och kan inte casta mer.",

        element = "air",
        type = "creature",
        health = stat.health(19),
        attack = stat.new(5),
        level = 7,

        onCast = function(self, game)
            game.startSpellTargeting(self, function(targetCard)
                targetCard.onCast = nil
                return true
            end)
        end,
        onDamage = function(game, card, fromCard, amount)
            if fromCard ~= nil and fromCard.type == "spell" then
                card.health.change(-amount * 0.5)
            else
                card.health.change(-amount)
            end
        end,
        onAttack = function(self, game)
            local opposingCard = game.opposingCard(self)
            if opposingCard ~= nil and opposingCard.onCast ~= nil then
                game.damageCard(opposingCard, self, 3)
            end
        end
    }
end

function airCards.nymph()
    return {
        name = "Nymph",
        ability = "Owner receives 1 Air at the beginning of Owner's turn.",
        ability_sv = "Ägaren får 1 Luft i början av dennes tur.",

        element = "air",
        type = "creature",
        health = stat.health(14),
        attack = stat.new(1),
        level = 3,

        onTurnStart = function(self, game) game.owner(self).air.change(1) end
    }
end

function airCards.phoenix()
    return {
        name = "Phoenix",
        ability = "If Phoenix was killed by Fire spell or creature, rebirths with full health. Whenever owner casts Fire spell, heals self for 3.",
        ability_sv = "Om Phoenix slås ut av Eld (varelse eller spell) så återföds den med full hälsa. När ägaren spelar ut en Eld-spell så helar den sig med 3.",

        element = "air",
        type = "creature",
        health = stat.health(20),
        attack = stat.new(4),
        level = 6,

        onAnyCardPlayed = function(self, game, otherCard)
            if otherCard ~= nil and otherCard.type == "spell" and
                otherCard.element == "fire" and game.owner(self) ==
                game.owner(otherCard) then game.healCard(self, 3) end
        end,
        onCardDies = function(self, game)
            if self.damagedByFire then
                local owner = game.owner(self)
                local position = game.cardPositionOnBoard(self)
                if position ~= nil then
                    owner.board[position] = CreateCard("phoenix")
                end
            end
        end,
        onDamage = function(game, card, fromCard, amount)
            card.damagedByFire = fromCard ~= nil and fromCard.element == "fire"
            card.health.change(-amount)
        end
    }
end

function airCards.titan()
    return {
        name = "Titan",
        ability = "When summoned, enemy loses 3 Air. Titan's attack is increased by 1 for each Air creature in play.",
        ability_sv = "När Titan spelas ut förlorar fienden 3 Luft. Titans attack ökar med 1 för varje Luftvarelse i spel.",
        spell = "Casts Thunder Fist. All enemy Earth creatures suffer 3 damage. Owner loses 1 Air.",
        spell_sv = "Castar Thunder Fist. Alla fiendens Jordvarelser tar 3 skada. Ägaren förlorar 1 Luft.",

        element = "air",
        type = "creature",
        health = stat.health(28),
        attack = stat.new(7),
        level = 7,

        onPlay = function(self, game)
            local owner = game.owner(self)
            local enemy = game.opponent(owner)
            enemy.air.change(-3)

            local airCardsInPlay = function()
                return Count(game.allCardsOnBoard(), function(card)
                    return card ~= nil and card ~= self and card.element ==
                               "air" and card.type == "creature"
                end)
            end

            self.attack = stat.derived(stat.new(7), function(baseValue)
                return baseValue + airCardsInPlay()
            end)
        end,
        canCast = function(self, game)
            return game.owner(self).air.canChange(-1)
        end,
        onCast = function(self, game)
            local enemyEarthCards = Where(game.enemyCards(self), function(card)
                return
                    card ~= nil and card.type == "creature" and card.element ==
                        "earth"
            end)
            ForEach(enemyEarthCards,
                    function(card) game.damageCard(card, self, 3) end)
            game.owner(self).air.change(-1)
        end
    }
end

function airCards.zeus()
    local function damageWithBenefits(self, game, enemyCard, amount)
        local enemyWasAlive = not enemyCard.dying
        game.damageCard(enemyCard, self, amount)
        local enemyIsAlive = not enemyCard.dying
        if enemyWasAlive and not enemyIsAlive then
            game.owner(self).air.change(1)
        end
    end

    return {
        name = "Zeus",
        ability = "Owner receives 1 Air element for each enemy creature, killed by Zeus.",
        ability_sv = "Ägaren får 1 Luft för varje fiende Zeus slår ut.",
        spell = "Strikes Lightning into choosen creature. Costs 1 Air and inflicts 8 damage. Cannot strike creatures of level 7 and higher.",
        spell_sv = "Skjuter blixtar mot vald varelse. Kostar 1 Luft och orsakar 8 skada. Kan inte välja varelser som är level 7 eller högre.",

        element = "air",
        type = "creature",
        health = stat.health(18),
        attack = stat.new(3),
        level = 10,

        canCast = function(self, game)
            return game.owner(self).air.canChange(-1)
        end,
        onCast = function(self, game)
            game.startSpellTargeting(self, function(targetCard)
                if targetCard.level < 7 then
                    damageWithBenefits(self, game, targetCard, 8)
                    game.owner(self).air.change(-1)

                    return true
                end

                return false
            end)
        end,
        onAttack = function(self, game)
            local enemyCard = game.opposingCard(self)
            if enemyCard ~= nil then
                damageWithBenefits(self, game, enemyCard, self.attack.value())
                return false -- stop attack
            end
            return true -- keep attacking to attack player
        end
    }
end

function airCards.blackwind()
    return {
        name = "Blackwind",
        ability = "Winds away the highest level enemy creature.",
        ability_sv = "Blåser iväg den fiende som har högst level.",
        spell = "Perfect against high-level enemy creatures. One of the most useful spells.",
        spell_sv = "Perfekt mot hög-levelmonster. En av de mest användbara spells.",

        element = "air",
        type = "spell",
        level = 8,

        onPlay = function(self, game)
            local strongestEnemyCard = game.strongestCard(game.enemyCards(self))
            if strongestEnemyCard ~= nil then
                game.tryToDestroyCard(strongestEnemyCard, self)
            end
        end
    }
end

function airCards.lightning()
    return {
        name = "Lightning",
        ability = "Hits enemy for 6 health.",
        ability_sv = "Träffar motspelaren med 6 skada.",
        spell = "One uses power of Lightning for the final blow.",
        spell_sv = "Användbar för att sätta dödsstöten.",

        element = "air",
        type = "spell",
        level = 4,

        onPlay = function(self, game)
            local opponent = game.opponent(game.owner(self))
            game.damagePlayer(opponent, self, 6)
        end
    }
end

function airCards.plague()
    return {
        name = "Plague",
        ability = "Every creature on a field plagued - loses all hit points except one. Ignores all defences and modifiers.",
        ability_sv = "Varje varelse på spelplanen får pesten - förlorar all sin hälsa utom en. Ignorerar alla skydd och modifikationer.",
        spell = "None shall escape the Plague! Great lands burnt to dust where the plague passed.",
        spell_sv = "Ingen kommer undan pesten! Mäktiga länder förvandlas till aska där pesten drar fram.",

        element = "air",
        type = "spell",
        level = 12,

        onPlay = function(self, game)
            ForEach(game.allCardsOnBoard(), function(card)
                if card.health ~= nil then
                    card.health.change(1 - card.health.value())
                end
            end)
        end
    }
end

function airCards.spellbreaker()
    return {
        name = "Spell breaker",
        ability = "Owner's creatures become permanently immune to all damaging spells, spell effects, and poison.",
        ability_sv = "Ägarens varelser blir permanent immuna mot alla skadade spells, spelleffekter och gift.",
        spell = "Remember that your creatures can no longer be affected by Bless, Restructure and other good spell effects.",
        spell_sv = "Kom ihåg att dina varelser inte längre kan påverkas av Bless, Restructure och andra bra spelleffekter.",

        element = "air",
        type = "spell",
        level = 7,

        onPlay = function(self, game)
            ForEach(game.owner(self).board,
                    function(card) effects.add(card, "spellbreaker") end)
        end
    }
end

return airCards
