local gameModule = require "model/game"
require "view/playarea"
local buttons = require "buttons"
local stat = require "model/stat"
local bot = require "bot"

local duel = {}

local infoPanelWidth = 439
local playAreaWidth = 500 + infoPanelWidth + 5
local playAreaHeight = 500

function duel.calculateScaleToFit()
    -- local boardMargin = 15
    -- local fightWidth = 480
    -- local playAreaWidth = boardMargin + fightWidth + boardMargin + infoPanelWidth + boardMargin

    local screenWidth, screenHeight = love.graphics.getDimensions()

    local verticalScale = screenHeight / playAreaHeight
    local horizontalScale = screenWidth / playAreaWidth

    return math.min(verticalScale, horizontalScale)
end

local function easeOutSine(x) return math.sin((x * math.pi) / 2); end
local function easeOutCubic(x) return 1 - math.pow(1 - x, 3) end
local endEase = easeOutCubic

local function drawEndScreen(progress, winner, loser)
    if not progress then return end
    love.graphics.push()

    local screenWidth, screenHeight = love.graphics.inverseTransformPoint(
                                          love.graphics.getDimensions())
    local height = assets.end_background:getHeight()
    local width = assets.end_background:getWidth()
    local from = screenHeight
    local movement = screenHeight / 2 + height / 2 -- screenHeight / 2 + height / 2

    -- CenterOnBox(assets.end_background:getWidth(),
    --             assets.end_background:getHeight())
    love.graphics.translate(screenWidth / 2 - width / 2,
                            from - endEase(progress.completion()) * movement)
    love.graphics.draw(assets.end_background, 0, 0)

    love.graphics.draw(assets[winner.avatar], 40, 40)
    love.graphics.print(winner.name .. " segrar!", 95, 50)

    love.graphics.draw(assets[loser.avatar], 40, 90)
    love.graphics.print(loser.name .. " tar disken!", 95, 100)

    love.graphics.pop()
end

local humanController = {}
humanController.new = function()
    local self = {}
    self.update = function(dt) end
    self.keypressed = function(key)
        TestHandKeys(key, events)
        TestBoardKeys(key, events)

        if key == "space" then
            table.insert(events, function(game) game.endTurn() end)
        end
    end
    return self
end

function duel.new(onExitToMenu, requireSpellSlot, player1, player2)
    local self = {}

    local handleEvent = function(event, payload)
        if event == "end" then
            self.winner = payload.winner
            self.loser = payload.loser
            self.endAnimation = stat.progress(3)
        end
        if assets[event] ~= nil then love.audio.play(assets[event]) end
    end

    local screenWidth, screenHeight = love.graphics.getDimensions()

    local xOffset = (screenWidth / scale - playAreaWidth) / 2
    local yOffset = (screenHeight / scale - playAreaHeight) / 2

    self.game = gameModule.new(handleEvent, requireSpellSlot, player1.deck,
                               player2.deck)
    local controller = function(player)
        return If(player.bot, bot.new(self.game), humanController.new())
    end
    self.game.player1().controller = controller(player1)
    self.game.player2().controller = controller(player2)

    local soundtrack = assets.soundtrack

    local stopSoundAndExit = function()
        onExitToMenu()
        if soundtrack then soundtrack:stop() end
    end

    local closeButton = buttons.closeButton.new(0, 0, stopSoundAndExit)
    local svButton =
        buttons.newLabelButton("SV", function() Language = "en" end)
    local enButton =
        buttons.newLabelButton("EN", function() Language = "sv" end)
    local function langButton()
        return If(Language == "sv", svButton, enButton)
    end

    function self.draw()
        if mouse.clicked and self.game.isTargetingSpell() then
            table.insert(events, function()
                self.game.cancelTargeting()
            end)
        end

        love.graphics.push()

        local screenWidth, screenHeight =
            love.graphics.inverseTransformPoint(love.graphics.getDimensions())

        love.graphics.translate(screenWidth - assets.close:getWidth() - 15, 15)
        closeButton.draw()
        love.graphics.translate(-40, 6)
        langButton().draw()
        love.graphics.pop()

        love.graphics.push()
        love.graphics.translate(xOffset, yOffset)
        DrawPlayArea(self.game, events)
        love.graphics.pop()

        drawEndScreen(self.endAnimation, self.winner, self.loser)

        if mouse.clicked and self.game.ended() and self.endAnimation and
            self.endAnimation.complete() then
            table.insert(events, function() self.endAnimation = nil end)
        end
    end

    function self.update(dt)
        if soundtrack and not soundtrack:isPlaying() then
            soundtrack:play()
        end

        self.game.update(dt)
        self.game.currentPlayer().controller.update(dt)

        if self.endAnimation then self.endAnimation.step(dt) end

        local cursor = love.mouse.getCursor()
        if self.game.isTargetingSpell() and cursor ~= targetCursor then
            love.mouse.setCursor(targetCursor)
        elseif not self.game.isTargetingSpell() and cursor ~= pointerCursor then
            love.mouse.setCursor(pointerCursor)
        end

        local event = events[#events]
        if event then event(self.game) end
        events = {}
    end

    function self.keypressed(key, scancode, isrepeat)
        self.game.currentPlayer().controller.keypressed(key)
    end

    -- handleEvent("end",
    --             {winner = self.game.player1(), loser = self.game.player2()})

    return self
end

return duel
