local utf8 = require "utf8"
local buttons = require "buttons"
local game = require "model/game"
require "model/cards"
require "view/drawcards"
require "view/infopanel"

local builder = {}

function builder.new(deck, onSave, onExit)
    local self = {}

    local screenWidthUnscaled, screenHeightUnscaled =
        love.graphics.getDimensions()
    local screenWidth, screenHeight = screenWidthUnscaled / scale,
                                      screenHeightUnscaled / scale

    local closeButtonX = screenWidth - buttons.closeButton.getWidth() - 5
    local closeButton = buttons.closeButton.new(closeButtonX, 5, onExit)

    local saveButton = buttons.saveButton.new(closeButtonX - 115, 5, onSave)

    local cards = AllCards()
    local game = game.new(function() end)

    local cellSize = 97

    local xMargin, yMarginTop, yMarginBottom = screenWidth * 0.05, 90,
                                               screenHeight * 0.05

    local cardsByElement =
        GroupBy(cards, function(card) return card.element end)
    local count = function(cards) return #cards end
    local maxNumberOfCardsInAnElement = Max(Select(cardsByElement, count))
    local neededWidth = maxNumberOfCardsInAnElement * cellSize + 2 * xMargin
    local neededHeigth = #Array(cardsByElement) * cellSize
    local cardScale = math.min(screenWidth / neededWidth, (screenHeight -
                                   yMarginTop - yMarginBottom) / neededHeigth)

    local function drawCards(cards)
        for i, card in ipairs(cards) do
            local column = i - 1
            local x = column * cellSize
            DrawCard({}, game, card, x, 0, If(deck.cards[card.name], 1, 0.6))

            if MouseOver(x, 0, cellSize, cellSize) then
                self.over = card
            end
            if ClickedInside(x, 0, cellSize, cellSize) then
                table.insert(events, function()
                    deck.cards[card.name] = not deck.cards[card.name]
                end)
            end
        end
    end

    local function translateToStartOfCards()
        love.graphics.scale(cardScale, cardScale)

        local screenWidth, _ = love.graphics.inverseTransformPoint(
                                   love.graphics.getDimensions())

        love.graphics.translate(xMargin, yMarginTop)
        love.graphics.translate((screenWidth - neededWidth) / 2, 0)
    end

    local function drawAllCards()
        love.graphics.push("all")
        translateToStartOfCards()

        for _, elementCards in pairs(cardsByElement) do
            elementCards = Array(elementCards)
            local selectedCount = Count(elementCards, function(card)
                return deck.cards[card.name]
            end)
            love.graphics.print(selectedCount, -20, 20)
            drawCards(elementCards)
            love.graphics.translate(0, cellSize)
        end
        love.graphics.pop()
    end

    local function drawInfoPanel()
        if not self.over then return end

        love.graphics.push("all")

        translateToStartOfCards()
        local offset = cellSize * 0.4
        local overX, overY = MousePosition()
        local width, height = 440, 270
        local drawX, drawY = overX - width / 2, overY + offset
        local screenWidth, screenHeight =
            love.graphics.inverseTransformPoint(love.graphics.getDimensions())

        if drawX < 0 then drawX = 0 end
        if drawX + width > screenWidth then drawX = screenWidth - width end

        if drawY + height > screenHeight then
            drawY = drawY - height - cellSize
        end

        DrawInfoPanel(drawX, drawY, game, assets, self.over)

        love.graphics.pop()
    end

    local function drawStats()
        local cardCount = Count(deck.cards,
                                function(selected) return selected end)
        love.graphics.print("Cards: " .. cardCount .. " / 24", screenWidth / 2,
                            15)
    end

    function self.draw()
        love.graphics.print(deck.name, 15, 15)
        closeButton.draw()
        saveButton.draw()
        drawAllCards()
        drawInfoPanel()
        drawStats()
    end

    function self.update(dt)
        self.over = nil

        local event = events[#events]
        if event then event() end
        events = {}
    end

    function self.keypressed(key, scancode, isrepeat)
        if key == "backspace" then
            -- get the byte offset to the last UTF-8 character in the string.
            local byteoffset = utf8.offset(deck.name, -1)

            if byteoffset then
                -- remove the last UTF-8 character.
                -- string.sub operates on bytes rather than UTF-8 characters, so we couldn't do string.sub(text, 1, -2).
                deck.name = string.sub(deck.name, 1, byteoffset - 1)
            end
        elseif utf8.len(key) == 1 then
            deck.name = deck.name .. key
        end
    end

    return self
end

return builder
